# Copyright Spack Project Developers. See COPYRIGHT file for details.
#
# SPDX-License-Identifier: (Apache-2.0 OR MIT)

# NOTE: spack-completion.fish is auto-generated by:
#
#   $ spack commands --aliases --format=fish
#       --header=fish/spack-completion.fish --update=spack-completion.fish
#
# Please do not manually modify this file.

# Check fish version before proceeding
set -l fish_version (string split '.' $FISH_VERSION)
if test $fish_version[1] -lt 3
    if test $fish_version[1] -eq 3
        and test $fish_version[2] -lt 2
        echo 'Fish version is older than 3.2.0. Some completion features may not work'
        set -g __fish_spack_force_files
    else
        echo 'This script requires fish version 3.0 or later'
        exit 1
    end
else
    set -g __fish_spack_force_files -F
end

# The following global variables are used as a cache of `__fish_spack_argparse`

# Cached command line
set -g __fish_spack_argparse_cache_line
# Parsed command
set -g __fish_spack_argparse_command
# Remaining arguments
set -g __fish_spack_argparse_argv
# Return value
set -g __fish_spack_argparse_return

# Spack command generates an optspec variable $__fish_spack_optspecs_<command>.
# We check if this command exists, and echo the optspec variable name.
function __fish_spack_get_optspecs -d 'Get optspecs of spack command'
    # Convert arguments to replace ' ' and '-' by '_'
    set -l cmd_var (string replace -ra -- '[ -]' '_' $argv | string join '_')
    # Set optspec variable name
    set -l optspecs_var __fish_spack_optspecs_$cmd_var
    # Query if variable $$optspecs_var exists
    set -q $optspecs_var; or return 1
    # If it exists, echo all optspecs line by line.
    # String join returns 1 if no join was performed, so we return 0 in such case.
    string join \n $$optspecs_var; or return 0
end

# Parse command-line arguments, save results to global variables,
# and add found flags to __fish_spack_flag_<flag>.
# Returns 1 if help flag is found.
function __fish_spack_argparse
    # Figure out if the current invocation already has a command.
    set -l args $argv
    set -l commands

    # Return cached result if arguments haven't changed
    if test "$__fish_spack_argparse_cache_line" = "$args"
        return $__fish_spack_argparse_return
    end

    # Clear all flags found in last run
    set -g | string replace -rf -- '^(__fish_spack_flag_\w+)(.*?)$' 'set -ge $1' | source

    # Set default return value to 0, indicating success
    set -g __fish_spack_argparse_return 0
    # Set command line to current arguments
    set -g __fish_spack_argparse_cache_line $argv

    # Recursively check arguments for commands
    while set -q args[1]
        # Get optspecs of current command
        set -l optspecs (__fish_spack_get_optspecs $commands $args[1])
        or break

        # If command exists, shift arguments
        set -a commands $args[1]
        set -e args[1]

        # If command has no arguments, continue
        set -q optspecs[1]; or continue

        # Parse arguments. Set variable _flag_<flag> if flag is found.
        # We find all these variables and set them to the global variable __fish_spack_flag_<flag>.
        argparse -i -s $optspecs -- $args 2>/dev/null; or break
        set -l | string replace -rf -- '^(_flag_.*)$' 'set -g __fish_spack$1' | source

        # Set args to not parsed arguments
        set args $argv

        # If command has help flag, we don't need to parse more so short circuit
        if set -q _flag_help
            set -g __fish_spack_argparse_return 1
            break
        end
    end

    # Set cached variables
    set -g __fish_spack_argparse_command $commands
    set -g __fish_spack_argparse_argv $args

    return $__fish_spack_argparse_return
end

# Check if current commandline's command is "spack $argv"
function __fish_spack_using_command
    set -l line (commandline -opc)
    __fish_spack_argparse $line; or return 1

    set -p argv spack
    test "$__fish_spack_argparse_command" = "$argv"
end

# Check if current commandline's command is "spack $argv[2..-1]",
# and cursor is at $argv[1]-th positional argument
function __fish_spack_using_command_pos
    __fish_spack_using_command $argv[2..-1]
    or return

    test (count $__fish_spack_argparse_argv) -eq $argv[1]
end

function __fish_spack_using_command_pos_remainder
    __fish_spack_using_command $argv[2..-1]
    or return

    test (count $__fish_spack_argparse_argv) -ge $argv[1]
end

# Helper functions for subcommands

function __fish_spack_bootstrap_names
    if set -q __fish_spack_flag_scope
        spack bootstrap list --scope $__fish_spack_flag_scope | string replace -rf -- '^Name: (\w+).*?$' '$1'
    else
        spack bootstrap list | string replace -rf -- '^Name: (\w+).*?$' '$1'
    end
end

# Reference: sudo's fish completion
function __fish_spack_build_env_spec
    set token (commandline -opt)

    set -l index (contains -- -- $__fish_spack_argparse_argv)
    if set -q index[1]
        __fish_complete_subcommand --commandline $__fish_spack_argparse_argv[(math $index + 1)..-1]
    else if set -q __fish_spack_argparse_argv[1]
        __fish_complete_subcommand --commandline "$__fish_spack_argparse_argv[2..-1] $token"
    else
        __fish_spack_specs
    end
end

function __fish_spack_commands
    spack commands
end

function __fish_spack_colon_path
    set token (string split -rm1 ':' (commandline -opt))

    if test (count $token) -lt 2
        __fish_complete_path $token[1]
    else
        __fish_complete_path $token[2] | string replace -r -- '^' "$token[1]:"
    end
end

function __fish_spack_config_sections
    if set -q __fish_spack_flag_scope
        spack config --scope $__fish_spack_flag_scope list | string split ' '
    else
        spack config list | string split ' '
    end
end

function __fish_spack_environments
    string trim (spack env list)
end

function __fish_spack_extensions
    # Skip optional flags, or it will be really slow
    string match -q -- '-*' (commandline -opt)
    and return

    comm -1 -2 (spack extensions | string trim | psub) (__fish_spack_installed_packages | sort | psub)
end

function __fish_spack_gpg_keys
    spack gpg list
end

function __fish_spack_installed_compilers
    spack compilers | grep -v '^[=-]\|^$'
end

function __fish_spack_installed_packages
    spack find --no-groups --format '{name}' | uniq
end

function __fish_spack_installed_specs
    # Try match local hash first
    __fish_spack_installed_specs_id
    and return

    spack find --no-groups --format '{name}@{version}'
end

function __fish_spack_installed_specs_id
    set -l token (commandline -opt)
    string match -q -- '/*' $token
    or return 1

    spack find --format '/{hash:7}'\t'{name}{@version}'
end

function __fish_spack_git_rev
    type -q __fish_git_ranges
    and __fish_git_ranges
end

function __fish_spack_mirrors
    spack mirror list | awk {'printf ("%s\t%s", $1, $2)'}
end

function __fish_spack_package_versions
    string trim (spack versions $argv)
end

function __fish_spack_packages
    spack list
end

function __fish_spack_pkg_packages
    spack pkg list
end

function __fish_spack_providers
    string trim (spack providers | grep -v '^$')
end

function __fish_spack_repos
    spack repo list --names
end

function __fish_spack_scopes
    # TODO: how to list all scopes?
    set -l scope system site user defaults
    set -l platform cray darwin linux test

    string join \n $scope
end

function __fish_spack_specs
    set -l token (commandline -opt)

    # Complete compilers
    if string match -rq -- '^(?<pre>.*%)[\w-]*(@[\w\.+~-]*)?$' $token
        __fish_spack_installed_compilers | string replace -r -- '^' "$pre"
        return
    end

    # Try to complete spec version
    # Currently we can only match '@' after a package name
    set -l package

    # Match ^ following package name
    if string match -rq -- '^(?<pre>.*?\^)[\w\.+~-]*$' $token
        # Package name is the nearest, assuming first character is always a letter or digit
        set packages (string match -ar -- '^[\w-]+' $__fish_spack_argparse_argv $token)
        set package $packages[-1]

        if test -n "$package"
            spack dependencies $package | string replace -r -- '^' "$pre"
            return
        end
    end

    # Match @ following package name
    if string match -rq -- '^(?<pre>.*?\^?(?<packages>[\w\.+~-]*)@)[\w\.]*$' $token
        set package $packages[-1]

        # Matched @ starting at next token
        if test -z "$package"
            string match -arq -- '(^|\^)(?<inners>[\w\.+~-]*)$' $__fish_spack_argparse_argv[-1]
            if test -n "$inners[1]"
                set package $inners[-1]
            end
        end
    end

    # Complete version if package found
    if test -n "$package"
        # Only list safe versions for speed
        string trim (spack versions --safe $package) | string replace -r -- '^' "$pre"
        return
    end

    # Else complete package name
    __fish_spack_installed_packages | string replace -r -- '$' \t"installed"
    spack list
end

function __fish_spack_specs_or_id
    # Try to match local hash first
    __fish_spack_installed_specs_id
    and return

    __fish_spack_specs
end

function __fish_spack_tags
    string trim (spack tags)
end

function __fish_spack_tests
    spack test list | grep -v '^[=-]'
end

function __fish_spack_unit_tests
    # Skip optional flags, or it will be really slow
    string match -q -- '-*' (commandline -opt)
    and return

    spack unit-test -l
end

function __fish_spack_yamls
    # Trim flag from current token
    string match -rq -- '(?<pre>-.)?(?<token>.*)' (commandline -opt)

    if test -n "$token"
        find $token* -type f '(' -iname '*.yaml' -or -iname '*.yml' ')'
    else
        find -maxdepth 2 -type f '(' -iname '*.yaml' -or -iname '*.yml' ')' | cut -c 3-
    end
end

# Reset existing completions
complete -c spack --erase

# Spack commands
#
# Everything below here is auto-generated.

# spack
set -g __fish_spack_optspecs_spack h/help H/all-help color= c/config= C/config-scope= d/debug timestamp pdb e/env= D/env-dir= E/no-env use-env-repo k/insecure l/enable-locks L/disable-locks m/mock b/bootstrap p/profile sorted-profile= lines= v/verbose stacktrace t/backtrace V/version print-shell-vars=
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a add -d 'add a spec to an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a arch -d 'print architecture information about this machine'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a audit -d 'audit configuration files, packages, etc.'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a blame -d 'show contributors to packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a bootstrap -d 'manage bootstrap configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a build-env -d 'run a command in a spec'"'"'s install environment, or dump its environment to screen or file'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a buildcache -d 'create, download and install binary packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a cd -d 'cd to spack directories in the shell'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a change -d 'change an existing spec in an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a checksum -d 'checksum available versions of a package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a ci -d 'manage continuous integration pipelines'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a clean -d 'remove temporary build files and/or downloaded archives'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a commands -d 'list available spack commands'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a compiler -d 'manage compilers'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a compilers -d 'list available compilers'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a concretize -d 'concretize an environment and write a lockfile'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a concretise -d 'concretize an environment and write a lockfile'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a config -d 'get and set configuration options'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a containerize -d 'creates recipes to build images for different container runtimes'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a containerise -d 'creates recipes to build images for different container runtimes'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a create -d 'create a new package file'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a debug -d 'debugging commands for troubleshooting Spack'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a deconcretize -d 'remove specs from the concretized lockfile of an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a dependencies -d 'show dependencies of a package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a dependents -d 'show packages that depend on another'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a deprecate -d 'replace one package with another via symlinks'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a dev-build -d 'developer build: build from code in current working directory'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a develop -d 'add a spec to an environment'"'"'s dev-build information'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a diff -d 'compare two specs'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a docs -d 'open spack documentation in a web browser'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a edit -d 'open package files in $EDITOR'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a env -d 'manage virtual environments'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a extensions -d 'list extensions for package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a external -d 'manage external packages in Spack configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a fetch -d 'fetch archives for packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a find -d 'list and search installed packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a gc -d 'remove specs that are now no longer needed'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a gpg -d 'handle GPG actions for spack'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a graph -d 'generate graphs of package dependency relationships'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a help -d 'get help on spack and its commands'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a info -d 'get detailed information on a particular package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a install -d 'build and install packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a license -d 'list and check license headers on files in spack'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a list -d 'list and search available packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a load -d 'add package to the user environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a location -d 'print out locations of packages and spack directories'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a log-parse -d 'filter errors and warnings from build logs'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a logs -d 'print out logs for packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a maintainers -d 'get information about package maintainers'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a make-installer -d 'generate Windows installer'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a mark -d 'mark packages as explicitly or implicitly installed'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a mirror -d 'manage mirrors (source and binary)'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a module -d 'generate/manage module files'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a patch -d 'patch expanded archive sources in preparation for install'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a pkg -d 'query packages associated with particular git revisions'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a providers -d 'list packages that provide a particular virtual package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a pydoc -d 'run pydoc from within spack'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a python -d 'launch an interpreter as spack would launch a command'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a reindex -d 'rebuild Spack'"'"'s package database'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a remove -d 'remove specs from an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a rm -d 'remove specs from an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a repo -d 'manage package source repositories'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a resource -d 'list downloadable resources (tarballs, repos, patches, etc.)'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a restage -d 'revert checked out package source code'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a solve -d 'concretize a specs using an ASP solver'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a spec -d 'show what would be installed, given a spec'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a stage -d 'expand downloaded archive in preparation for install'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a style -d 'runs source code style checks on spack'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a tags -d 'show package tags and associated packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a test -d 'run spack'"'"'s tests for an install'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a test-env -d 'run a command in a spec'"'"'s test environment, or dump its environment to screen or file'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a tutorial -d 'set up spack for our tutorial (WARNING: modifies config!)'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a undevelop -d 'remove specs from an environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a uninstall -d 'remove installed packages'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a unit-test -d 'run spack'"'"'s unit tests (wrapper around pytest)'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a unload -d 'remove package from the user environment'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a url -d 'debugging tool for url parsing'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a verify -d 'verify spack installations on disk'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a versions -d 'list available versions of a package'
complete -c spack -n '__fish_spack_using_command_pos 0 ' -f -a view -d 'project packages to a compact naming scheme on the filesystem'
complete -c spack -n '__fish_spack_using_command ' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command ' -s H -l all-help -f -a help
complete -c spack -n '__fish_spack_using_command ' -s H -l all-help -d 'show help for all commands (same as spack help --all)'
complete -c spack -n '__fish_spack_using_command ' -l color -r -f -a 'always never auto'
complete -c spack -n '__fish_spack_using_command ' -l color -r -d 'when to colorize output (default: auto)'
complete -c spack -n '__fish_spack_using_command ' -s c -l config -r -f -a config_vars
complete -c spack -n '__fish_spack_using_command ' -s c -l config -r -d 'add one or more custom, one off config settings'
complete -c spack -n '__fish_spack_using_command ' -s C -l config-scope -r -f -a config_scopes
complete -c spack -n '__fish_spack_using_command ' -s C -l config-scope -r -d 'add directory or environment as read-only configuration scope, without activating the environment.'
complete -c spack -n '__fish_spack_using_command ' -s d -l debug -f -a debug
complete -c spack -n '__fish_spack_using_command ' -s d -l debug -d 'write out debug messages'
complete -c spack -n '__fish_spack_using_command ' -l timestamp -f -a timestamp
complete -c spack -n '__fish_spack_using_command ' -l timestamp -d 'add a timestamp to tty output'
complete -c spack -n '__fish_spack_using_command ' -l pdb -f -a pdb
complete -c spack -n '__fish_spack_using_command ' -l pdb -d 'run spack under the pdb debugger'
complete -c spack -n '__fish_spack_using_command ' -s e -l env -r -f -a env
complete -c spack -n '__fish_spack_using_command ' -s e -l env -r -d 'run with a specific environment (see spack env)'
complete -c spack -n '__fish_spack_using_command ' -s D -l env-dir -r -f -a env_dir
complete -c spack -n '__fish_spack_using_command ' -s D -l env-dir -r -d 'run with an environment directory (ignore managed environments)'
complete -c spack -n '__fish_spack_using_command ' -s E -l no-env -f -a no_env
complete -c spack -n '__fish_spack_using_command ' -s E -l no-env -d 'run without any environments activated (see spack env)'
complete -c spack -n '__fish_spack_using_command ' -l use-env-repo -f -a use_env_repo
complete -c spack -n '__fish_spack_using_command ' -l use-env-repo -d 'when running in an environment, use its package repository'
complete -c spack -n '__fish_spack_using_command ' -s k -l insecure -f -a insecure
complete -c spack -n '__fish_spack_using_command ' -s k -l insecure -d 'do not check ssl certificates when downloading'
complete -c spack -n '__fish_spack_using_command ' -s l -l enable-locks -f -a locks
complete -c spack -n '__fish_spack_using_command ' -s l -l enable-locks -d 'use filesystem locking (default)'
complete -c spack -n '__fish_spack_using_command ' -s L -l disable-locks -f -a locks
complete -c spack -n '__fish_spack_using_command ' -s L -l disable-locks -d 'do not use filesystem locking (unsafe)'
complete -c spack -n '__fish_spack_using_command ' -s m -l mock -f -a mock
complete -c spack -n '__fish_spack_using_command ' -s m -l mock -d 'use mock packages instead of real ones'
complete -c spack -n '__fish_spack_using_command ' -s b -l bootstrap -f -a bootstrap
complete -c spack -n '__fish_spack_using_command ' -s b -l bootstrap -d 'use bootstrap configuration (bootstrap store, config, externals)'
complete -c spack -n '__fish_spack_using_command ' -s p -l profile -f -a spack_profile
complete -c spack -n '__fish_spack_using_command ' -s p -l profile -d 'profile execution using cProfile'
complete -c spack -n '__fish_spack_using_command ' -l sorted-profile -r -f -a sorted_profile
complete -c spack -n '__fish_spack_using_command ' -l sorted-profile -r -d 'profile and sort'
complete -c spack -n '__fish_spack_using_command ' -l lines -r -f -a lines
complete -c spack -n '__fish_spack_using_command ' -l lines -r -d 'lines of profile output or '"'"'all'"'"' (default: 20)'
complete -c spack -n '__fish_spack_using_command ' -s v -l verbose -f -a verbose
complete -c spack -n '__fish_spack_using_command ' -s v -l verbose -d 'print additional output during builds'
complete -c spack -n '__fish_spack_using_command ' -l stacktrace -f -a stacktrace
complete -c spack -n '__fish_spack_using_command ' -l stacktrace -d 'add stacktraces to all printed statements'
complete -c spack -n '__fish_spack_using_command ' -s t -l backtrace -f -a backtrace
complete -c spack -n '__fish_spack_using_command ' -s t -l backtrace -d 'always show backtraces for exceptions'
complete -c spack -n '__fish_spack_using_command ' -s V -l version -f -a version
complete -c spack -n '__fish_spack_using_command ' -s V -l version -d 'show version number and exit'
complete -c spack -n '__fish_spack_using_command ' -l print-shell-vars -r -f -a print_shell_vars
complete -c spack -n '__fish_spack_using_command ' -l print-shell-vars -r -d 'print info needed by setup-env.*sh'

# spack add
set -g __fish_spack_optspecs_spack_add h/help l/list-name=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 add' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command add' -s l -l list-name -r -f -a list_name
complete -c spack -n '__fish_spack_using_command add' -s l -l list-name -r -d 'name of the list to add specs to'

# spack arch
set -g __fish_spack_optspecs_spack_arch h/help g/generic-target known-targets family generic p/platform o/operating-system t/target f/frontend b/backend
complete -c spack -n '__fish_spack_using_command arch' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command arch' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command arch' -s g -l generic-target -f -a generic_target
complete -c spack -n '__fish_spack_using_command arch' -s g -l generic-target -d 'show the best generic target (deprecated)'
complete -c spack -n '__fish_spack_using_command arch' -l known-targets -f -a known_targets
complete -c spack -n '__fish_spack_using_command arch' -l known-targets -d 'show a list of all known targets and exit'
complete -c spack -n '__fish_spack_using_command arch' -l family -f -a family
complete -c spack -n '__fish_spack_using_command arch' -l family -d 'print generic ISA (x86_64, aarch64, ppc64le, ...)'
complete -c spack -n '__fish_spack_using_command arch' -l generic -f -a generic
complete -c spack -n '__fish_spack_using_command arch' -l generic -d 'print feature level (x86_64_v3, armv8.4a, ...)'
complete -c spack -n '__fish_spack_using_command arch' -s p -l platform -f -a platform
complete -c spack -n '__fish_spack_using_command arch' -s p -l platform -d 'print only the platform'
complete -c spack -n '__fish_spack_using_command arch' -s o -l operating-system -f -a operating_system
complete -c spack -n '__fish_spack_using_command arch' -s o -l operating-system -d 'print only the operating system'
complete -c spack -n '__fish_spack_using_command arch' -s t -l target -f -a target
complete -c spack -n '__fish_spack_using_command arch' -s t -l target -d 'print only the target'
complete -c spack -n '__fish_spack_using_command arch' -s f -l frontend -f -a frontend
complete -c spack -n '__fish_spack_using_command arch' -s f -l frontend -d 'print frontend (DEPRECATED)'
complete -c spack -n '__fish_spack_using_command arch' -s b -l backend -f -a backend
complete -c spack -n '__fish_spack_using_command arch' -s b -l backend -d 'print backend (DEPRECATED)'

# spack audit
set -g __fish_spack_optspecs_spack_audit h/help
complete -c spack -n '__fish_spack_using_command_pos 0 audit' -f -a configs -d 'audit configuration files'
complete -c spack -n '__fish_spack_using_command_pos 0 audit' -f -a externals -d 'check external detection in packages'
complete -c spack -n '__fish_spack_using_command_pos 0 audit' -f -a packages-https -d 'check https in packages'
complete -c spack -n '__fish_spack_using_command_pos 0 audit' -f -a packages -d 'audit package recipes'
complete -c spack -n '__fish_spack_using_command_pos 0 audit' -f -a list -d 'list available checks and exits'
complete -c spack -n '__fish_spack_using_command audit' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit' -s h -l help -d 'show this help message and exit'

# spack audit configs
set -g __fish_spack_optspecs_spack_audit_configs h/help
complete -c spack -n '__fish_spack_using_command audit configs' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit configs' -s h -l help -d 'show this help message and exit'

# spack audit externals
set -g __fish_spack_optspecs_spack_audit_externals h/help list
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 audit externals' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command audit externals' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit externals' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command audit externals' -l list -f -a list_externals
complete -c spack -n '__fish_spack_using_command audit externals' -l list -d 'if passed, list which packages have detection tests'

# spack audit packages-https
set -g __fish_spack_optspecs_spack_audit_packages_https h/help all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 audit packages-https' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command audit packages-https' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit packages-https' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command audit packages-https' -l all -f -a check_all
complete -c spack -n '__fish_spack_using_command audit packages-https' -l all -d 'audit all packages'

# spack audit packages
set -g __fish_spack_optspecs_spack_audit_packages h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 audit packages' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command audit packages' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit packages' -s h -l help -d 'show this help message and exit'

# spack audit list
set -g __fish_spack_optspecs_spack_audit_list h/help
complete -c spack -n '__fish_spack_using_command audit list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command audit list' -s h -l help -d 'show this help message and exit'

# spack blame
set -g __fish_spack_optspecs_spack_blame h/help t/time p/percent g/git json
complete -c spack -n '__fish_spack_using_command_pos 0 blame' $__fish_spack_force_files -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command blame' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command blame' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command blame' -s t -l time -f -a view
complete -c spack -n '__fish_spack_using_command blame' -s t -l time -d 'sort by last modification date (default)'
complete -c spack -n '__fish_spack_using_command blame' -s p -l percent -f -a view
complete -c spack -n '__fish_spack_using_command blame' -s p -l percent -d 'sort by percent of code'
complete -c spack -n '__fish_spack_using_command blame' -s g -l git -f -a view
complete -c spack -n '__fish_spack_using_command blame' -s g -l git -d 'show git blame output instead of summary'
complete -c spack -n '__fish_spack_using_command blame' -l json -f -a json
complete -c spack -n '__fish_spack_using_command blame' -l json -d 'output blame as machine-readable json records'

# spack bootstrap
set -g __fish_spack_optspecs_spack_bootstrap h/help
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a now -d 'Spack ready, right now!'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a status -d 'get the status of Spack'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a enable -d 'enable bootstrapping'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a disable -d 'disable bootstrapping'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a reset -d 'reset bootstrapping configuration to Spack defaults'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a root -d 'get/set the root bootstrap directory'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a list -d 'list all the sources of software to bootstrap Spack'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a add -d 'add a new source for bootstrapping'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a remove -d 'remove a bootstrapping source'
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap' -f -a mirror -d 'create a local mirror to bootstrap Spack'
complete -c spack -n '__fish_spack_using_command bootstrap' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap' -s h -l help -d 'show this help message and exit'

# spack bootstrap now
set -g __fish_spack_optspecs_spack_bootstrap_now h/help dev
complete -c spack -n '__fish_spack_using_command bootstrap now' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap now' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap now' -l dev -f -a dev
complete -c spack -n '__fish_spack_using_command bootstrap now' -l dev -d 'bootstrap dev dependencies too'

# spack bootstrap status
set -g __fish_spack_optspecs_spack_bootstrap_status h/help optional dev
complete -c spack -n '__fish_spack_using_command bootstrap status' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap status' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap status' -l optional -f -a optional
complete -c spack -n '__fish_spack_using_command bootstrap status' -l optional -d 'show the status of rarely used optional dependencies'
complete -c spack -n '__fish_spack_using_command bootstrap status' -l dev -f -a dev
complete -c spack -n '__fish_spack_using_command bootstrap status' -l dev -d 'show the status of dependencies needed to develop Spack'

# spack bootstrap enable
set -g __fish_spack_optspecs_spack_bootstrap_enable h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap enable' -f -a '(__fish_spack_bootstrap_names)'
complete -c spack -n '__fish_spack_using_command bootstrap enable' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap enable' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap enable' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command bootstrap enable' -l scope -r -d 'configuration scope to read/modify'

# spack bootstrap disable
set -g __fish_spack_optspecs_spack_bootstrap_disable h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap disable' -f -a '(__fish_spack_bootstrap_names)'
complete -c spack -n '__fish_spack_using_command bootstrap disable' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap disable' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap disable' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command bootstrap disable' -l scope -r -d 'configuration scope to read/modify'

# spack bootstrap reset
set -g __fish_spack_optspecs_spack_bootstrap_reset h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command bootstrap reset' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap reset' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap reset' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command bootstrap reset' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack bootstrap root
set -g __fish_spack_optspecs_spack_bootstrap_root h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap root' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command bootstrap root' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap root' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap root' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command bootstrap root' -l scope -r -d 'configuration scope to read/modify'

# spack bootstrap list
set -g __fish_spack_optspecs_spack_bootstrap_list h/help scope=
complete -c spack -n '__fish_spack_using_command bootstrap list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap list' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command bootstrap list' -l scope -r -d 'configuration scope to read/modify'

# spack bootstrap add
set -g __fish_spack_optspecs_spack_bootstrap_add h/help scope= trust
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap add' -f -a '(__fish_spack_bootstrap_names)'
complete -c spack -n '__fish_spack_using_command_pos 1 bootstrap add' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command bootstrap add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap add' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command bootstrap add' -l scope -r -d 'configuration scope to read/modify'
complete -c spack -n '__fish_spack_using_command bootstrap add' -l trust -f -a trust
complete -c spack -n '__fish_spack_using_command bootstrap add' -l trust -d 'enable the source immediately upon addition'

# spack bootstrap remove
set -g __fish_spack_optspecs_spack_bootstrap_remove h/help
complete -c spack -n '__fish_spack_using_command_pos 0 bootstrap remove' -f -a '(__fish_spack_bootstrap_names)'
complete -c spack -n '__fish_spack_using_command bootstrap remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap remove' -s h -l help -d 'show this help message and exit'

# spack bootstrap mirror
set -g __fish_spack_optspecs_spack_bootstrap_mirror h/help binary-packages dev

complete -c spack -n '__fish_spack_using_command bootstrap mirror' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command bootstrap mirror' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command bootstrap mirror' -l binary-packages -f -a binary_packages
complete -c spack -n '__fish_spack_using_command bootstrap mirror' -l binary-packages -d 'download public binaries in the mirror'
complete -c spack -n '__fish_spack_using_command bootstrap mirror' -l dev -f -a dev
complete -c spack -n '__fish_spack_using_command bootstrap mirror' -l dev -d 'download dev dependencies too'

# spack build-env
set -g __fish_spack_optspecs_spack_build_env h/help clean dirty f/force U/fresh reuse fresh-roots deprecated dump= pickle=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 build-env' -f -a '(__fish_spack_build_env_spec)'
complete -c spack -n '__fish_spack_using_command build-env' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command build-env' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command build-env' -l clean -f -a dirty
complete -c spack -n '__fish_spack_using_command build-env' -l clean -d 'unset harmful variables in the build environment (default)'
complete -c spack -n '__fish_spack_using_command build-env' -l dirty -f -a dirty
complete -c spack -n '__fish_spack_using_command build-env' -l dirty -d 'preserve user environment in spack'"'"'s build environment (danger!)'
complete -c spack -n '__fish_spack_using_command build-env' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command build-env' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command build-env' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command build-env' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command build-env' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command build-env' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command build-env' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command build-env' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command build-env' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command build-env' -l deprecated -d 'allow concretizer to select deprecated versions'
complete -c spack -n '__fish_spack_using_command build-env' -l dump -r -f -a dump
complete -c spack -n '__fish_spack_using_command build-env' -l dump -r -d 'dump a source-able environment to FILE'
complete -c spack -n '__fish_spack_using_command build-env' -l pickle -r -f -a pickle
complete -c spack -n '__fish_spack_using_command build-env' -l pickle -r -d 'dump a pickled source-able environment to FILE'

# spack buildcache
set -g __fish_spack_optspecs_spack_buildcache h/help
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a push -d 'create a binary package and push it to a mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a create -d 'create a binary package and push it to a mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a install -d 'install from a binary package'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a list -d 'list binary packages available from mirrors'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a keys -d 'get public keys available on mirrors'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a check -d 'check specs against remote binary mirror(s) to see if any need to be rebuilt'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a download -d 'download buildcache entry from a remote mirror to local folder'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a prune -d 'prune stale buildcache entries from the mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a save-specfile -d 'get full spec for dependencies and write them to files in the specified output directory'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a sync -d 'sync binaries (and associated metadata) from one mirror to another'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a update-index -d 'update a buildcache index'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a rebuild-index -d 'update a buildcache index'
complete -c spack -n '__fish_spack_using_command_pos 0 buildcache' -f -a migrate -d 'perform in-place binary mirror migration (2 to 3)'
complete -c spack -n '__fish_spack_using_command buildcache' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache' -s h -l help -d 'show this help message and exit'

# spack buildcache push
set -g __fish_spack_optspecs_spack_buildcache_push h/help f/force u/unsigned signed k/key= update-index only= with-build-dependencies without-build-dependencies fail-fast base-image= t/tag= private j/jobs=
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 buildcache push' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command buildcache push' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache push' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache push' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command buildcache push' -s f -l force -d 'overwrite tarball if it exists'
complete -c spack -n '__fish_spack_using_command buildcache push' -l unsigned -s u -f -a signed
complete -c spack -n '__fish_spack_using_command buildcache push' -l unsigned -s u -d 'push unsigned buildcache tarballs'
complete -c spack -n '__fish_spack_using_command buildcache push' -l signed -f -a signed
complete -c spack -n '__fish_spack_using_command buildcache push' -l signed -d 'push signed buildcache tarballs'
complete -c spack -n '__fish_spack_using_command buildcache push' -l key -s k -r -f -a key
complete -c spack -n '__fish_spack_using_command buildcache push' -l key -s k -r -d 'key for signing'
complete -c spack -n '__fish_spack_using_command buildcache push' -l update-index -l rebuild-index -f -a update_index
complete -c spack -n '__fish_spack_using_command buildcache push' -l update-index -l rebuild-index -d 'regenerate buildcache index after building package(s)'
complete -c spack -n '__fish_spack_using_command buildcache push' -l only -r -f -a 'package dependencies'
complete -c spack -n '__fish_spack_using_command buildcache push' -l only -r -d 'select the buildcache mode. The default is to build a cache for the package along with all its dependencies. Alternatively, one can decide to build a cache for only the package or only the dependencies'
complete -c spack -n '__fish_spack_using_command buildcache push' -l with-build-dependencies -f -a with_build_dependencies
complete -c spack -n '__fish_spack_using_command buildcache push' -l with-build-dependencies -d 'include build dependencies in the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache push' -l without-build-dependencies -f -a without_build_dependencies
complete -c spack -n '__fish_spack_using_command buildcache push' -l without-build-dependencies -d 'exclude build dependencies from the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache push' -l fail-fast -f -a fail_fast
complete -c spack -n '__fish_spack_using_command buildcache push' -l fail-fast -d 'stop pushing on first failure (default is best effort)'
complete -c spack -n '__fish_spack_using_command buildcache push' -l base-image -r -f -a base_image
complete -c spack -n '__fish_spack_using_command buildcache push' -l base-image -r -d 'specify the base image for the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache push' -l tag -s t -r -f -a tag
complete -c spack -n '__fish_spack_using_command buildcache push' -l tag -s t -r -d 'when pushing to an OCI registry, tag an image containing all root specs and their runtime dependencies'
complete -c spack -n '__fish_spack_using_command buildcache push' -l private -f -a private
complete -c spack -n '__fish_spack_using_command buildcache push' -l private -d 'for a private mirror, include non-redistributable packages'
complete -c spack -n '__fish_spack_using_command buildcache push' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command buildcache push' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack buildcache create
set -g __fish_spack_optspecs_spack_buildcache_create h/help f/force u/unsigned signed k/key= update-index only= with-build-dependencies without-build-dependencies fail-fast base-image= t/tag= private j/jobs=
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 buildcache create' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command buildcache create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache create' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command buildcache create' -s f -l force -d 'overwrite tarball if it exists'
complete -c spack -n '__fish_spack_using_command buildcache create' -l unsigned -s u -f -a signed
complete -c spack -n '__fish_spack_using_command buildcache create' -l unsigned -s u -d 'push unsigned buildcache tarballs'
complete -c spack -n '__fish_spack_using_command buildcache create' -l signed -f -a signed
complete -c spack -n '__fish_spack_using_command buildcache create' -l signed -d 'push signed buildcache tarballs'
complete -c spack -n '__fish_spack_using_command buildcache create' -l key -s k -r -f -a key
complete -c spack -n '__fish_spack_using_command buildcache create' -l key -s k -r -d 'key for signing'
complete -c spack -n '__fish_spack_using_command buildcache create' -l update-index -l rebuild-index -f -a update_index
complete -c spack -n '__fish_spack_using_command buildcache create' -l update-index -l rebuild-index -d 'regenerate buildcache index after building package(s)'
complete -c spack -n '__fish_spack_using_command buildcache create' -l only -r -f -a 'package dependencies'
complete -c spack -n '__fish_spack_using_command buildcache create' -l only -r -d 'select the buildcache mode. The default is to build a cache for the package along with all its dependencies. Alternatively, one can decide to build a cache for only the package or only the dependencies'
complete -c spack -n '__fish_spack_using_command buildcache create' -l with-build-dependencies -f -a with_build_dependencies
complete -c spack -n '__fish_spack_using_command buildcache create' -l with-build-dependencies -d 'include build dependencies in the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache create' -l without-build-dependencies -f -a without_build_dependencies
complete -c spack -n '__fish_spack_using_command buildcache create' -l without-build-dependencies -d 'exclude build dependencies from the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache create' -l fail-fast -f -a fail_fast
complete -c spack -n '__fish_spack_using_command buildcache create' -l fail-fast -d 'stop pushing on first failure (default is best effort)'
complete -c spack -n '__fish_spack_using_command buildcache create' -l base-image -r -f -a base_image
complete -c spack -n '__fish_spack_using_command buildcache create' -l base-image -r -d 'specify the base image for the buildcache'
complete -c spack -n '__fish_spack_using_command buildcache create' -l tag -s t -r -f -a tag
complete -c spack -n '__fish_spack_using_command buildcache create' -l tag -s t -r -d 'when pushing to an OCI registry, tag an image containing all root specs and their runtime dependencies'
complete -c spack -n '__fish_spack_using_command buildcache create' -l private -f -a private
complete -c spack -n '__fish_spack_using_command buildcache create' -l private -d 'for a private mirror, include non-redistributable packages'
complete -c spack -n '__fish_spack_using_command buildcache create' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command buildcache create' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack buildcache install
set -g __fish_spack_optspecs_spack_buildcache_install h/help f/force m/multiple u/unsigned o/otherarch
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 buildcache install' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command buildcache install' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache install' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache install' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command buildcache install' -s f -l force -d 'overwrite install directory if it exists'
complete -c spack -n '__fish_spack_using_command buildcache install' -s m -l multiple -f -a multiple
complete -c spack -n '__fish_spack_using_command buildcache install' -s m -l multiple -d 'allow all matching packages'
complete -c spack -n '__fish_spack_using_command buildcache install' -s u -l unsigned -f -a unsigned
complete -c spack -n '__fish_spack_using_command buildcache install' -s u -l unsigned -d 'install unsigned buildcache tarballs for testing'
complete -c spack -n '__fish_spack_using_command buildcache install' -s o -l otherarch -f -a otherarch
complete -c spack -n '__fish_spack_using_command buildcache install' -s o -l otherarch -d 'install specs from other architectures instead of default platform and OS'

# spack buildcache list
set -g __fish_spack_optspecs_spack_buildcache_list h/help l/long L/very-long N/namespaces v/variants a/allarch
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 buildcache list' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command buildcache list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache list' -s l -l long -f -a long
complete -c spack -n '__fish_spack_using_command buildcache list' -s l -l long -d 'show dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command buildcache list' -s L -l very-long -f -a very_long
complete -c spack -n '__fish_spack_using_command buildcache list' -s L -l very-long -d 'show full dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command buildcache list' -s N -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command buildcache list' -s N -l namespaces -d 'show fully qualified package names'
complete -c spack -n '__fish_spack_using_command buildcache list' -s v -l variants -f -a variants
complete -c spack -n '__fish_spack_using_command buildcache list' -s v -l variants -d 'show variants in output (can be long)'
complete -c spack -n '__fish_spack_using_command buildcache list' -s a -l allarch -f -a allarch
complete -c spack -n '__fish_spack_using_command buildcache list' -s a -l allarch -d 'list specs for all available architectures instead of default platform and OS'

# spack buildcache keys
set -g __fish_spack_optspecs_spack_buildcache_keys h/help i/install t/trust f/force
complete -c spack -n '__fish_spack_using_command buildcache keys' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache keys' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache keys' -s i -l install -f -a install
complete -c spack -n '__fish_spack_using_command buildcache keys' -s i -l install -d 'install Keys pulled from mirror'
complete -c spack -n '__fish_spack_using_command buildcache keys' -s t -l trust -f -a trust
complete -c spack -n '__fish_spack_using_command buildcache keys' -s t -l trust -d 'trust all downloaded keys'
complete -c spack -n '__fish_spack_using_command buildcache keys' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command buildcache keys' -s f -l force -d 'force new download of keys'

# spack buildcache check
set -g __fish_spack_optspecs_spack_buildcache_check h/help m/mirror-url= o/output-file= scope=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 buildcache check' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command buildcache check' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache check' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache check' -s m -l mirror-url -r -f -a mirror_url
complete -c spack -n '__fish_spack_using_command buildcache check' -s m -l mirror-url -r -d 'override any configured mirrors with this mirror URL'
complete -c spack -n '__fish_spack_using_command buildcache check' -s o -l output-file -r -f -a output_file
complete -c spack -n '__fish_spack_using_command buildcache check' -s o -l output-file -r -d 'file where rebuild info should be written'
complete -c spack -n '__fish_spack_using_command buildcache check' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command buildcache check' -l scope -r -d 'configuration scope containing mirrors to check'

# spack buildcache download
set -g __fish_spack_optspecs_spack_buildcache_download h/help s/spec= p/path=
complete -c spack -n '__fish_spack_using_command buildcache download' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache download' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache download' -s s -l spec -r -f -a spec
complete -c spack -n '__fish_spack_using_command buildcache download' -s s -l spec -r -d 'download built tarball for spec from mirror'
complete -c spack -n '__fish_spack_using_command buildcache download' -s p -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command buildcache download' -s p -l path -r -d 'path to directory where tarball should be downloaded'

# spack buildcache prune
set -g __fish_spack_optspecs_spack_buildcache_prune h/help dry-run

complete -c spack -n '__fish_spack_using_command buildcache prune' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache prune' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache prune' -l dry-run -f -a dry_run
complete -c spack -n '__fish_spack_using_command buildcache prune' -l dry-run -d 'do not actually delete anything from the buildcache, but log what would be deleted'

# spack buildcache save-specfile
set -g __fish_spack_optspecs_spack_buildcache_save_specfile h/help root-spec= s/specs= specfile-dir=
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -l root-spec -r -f -a root_spec
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -l root-spec -r -d 'root spec of dependent spec'
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -s s -l specs -r -f -a specs
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -s s -l specs -r -d 'list of dependent specs for which saved yaml is desired'
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -l specfile-dir -r -f -a specfile_dir
complete -c spack -n '__fish_spack_using_command buildcache save-specfile' -l specfile-dir -r -d 'path to directory where spec yamls should be saved'

# spack buildcache sync
set -g __fish_spack_optspecs_spack_buildcache_sync h/help manifest-glob=

complete -c spack -n '__fish_spack_using_command buildcache sync' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache sync' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache sync' -l manifest-glob -r -f -a manifest_glob
complete -c spack -n '__fish_spack_using_command buildcache sync' -l manifest-glob -r -d 'a quoted glob pattern identifying CI rebuild manifest files'

# spack buildcache update-index
set -g __fish_spack_optspecs_spack_buildcache_update_index h/help k/keys

complete -c spack -n '__fish_spack_using_command buildcache update-index' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache update-index' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache update-index' -s k -l keys -f -a keys
complete -c spack -n '__fish_spack_using_command buildcache update-index' -s k -l keys -d 'if provided, key index will be updated as well as package index'

# spack buildcache rebuild-index
set -g __fish_spack_optspecs_spack_buildcache_rebuild_index h/help k/keys

complete -c spack -n '__fish_spack_using_command buildcache rebuild-index' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache rebuild-index' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache rebuild-index' -s k -l keys -f -a keys
complete -c spack -n '__fish_spack_using_command buildcache rebuild-index' -s k -l keys -d 'if provided, key index will be updated as well as package index'

# spack buildcache migrate
set -g __fish_spack_optspecs_spack_buildcache_migrate h/help u/unsigned d/delete-existing y/yes-to-all

complete -c spack -n '__fish_spack_using_command buildcache migrate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s u -l unsigned -f -a unsigned
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s u -l unsigned -d 'Ignore signatures and do not resign, default is False'
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s d -l delete-existing -f -a delete_existing
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s d -l delete-existing -d 'Delete the previous layout, the default is to keep it.'
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command buildcache migrate' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack cd
set -g __fish_spack_optspecs_spack_cd h/help m/module-dir r/spack-root i/install-dir p/package-dir repo= s/stage-dir S/stages c/source-dir b/build-dir e/env= first
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 cd' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command cd' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command cd' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command cd' -s m -l module-dir -f -a module_dir
complete -c spack -n '__fish_spack_using_command cd' -s m -l module-dir -d 'spack python module directory'
complete -c spack -n '__fish_spack_using_command cd' -s r -l spack-root -f -a spack_root
complete -c spack -n '__fish_spack_using_command cd' -s r -l spack-root -d 'spack installation root'
complete -c spack -n '__fish_spack_using_command cd' -s i -l install-dir -f -a install_dir
complete -c spack -n '__fish_spack_using_command cd' -s i -l install-dir -d 'install prefix for spec (spec need not be installed)'
complete -c spack -n '__fish_spack_using_command cd' -s p -l package-dir -f -a package_dir
complete -c spack -n '__fish_spack_using_command cd' -s p -l package-dir -d 'directory enclosing a spec'"'"'s package.py file'
complete -c spack -n '__fish_spack_using_command cd' -l repo -l packages -s P -r -f -a repo
complete -c spack -n '__fish_spack_using_command cd' -l repo -l packages -s P -r -d 'package repository root (defaults to first configured repository)'
complete -c spack -n '__fish_spack_using_command cd' -s s -l stage-dir -f -a stage_dir
complete -c spack -n '__fish_spack_using_command cd' -s s -l stage-dir -d 'stage directory for a spec'
complete -c spack -n '__fish_spack_using_command cd' -s S -l stages -f -a stages
complete -c spack -n '__fish_spack_using_command cd' -s S -l stages -d 'top level stage directory'
complete -c spack -n '__fish_spack_using_command cd' -s c -l source-dir -f -a source_dir
complete -c spack -n '__fish_spack_using_command cd' -s c -l source-dir -d 'source directory for a spec (requires it to be staged first)'
complete -c spack -n '__fish_spack_using_command cd' -s b -l build-dir -f -a build_dir
complete -c spack -n '__fish_spack_using_command cd' -s b -l build-dir -d 'build directory for a spec (requires it to be staged first)'
complete -c spack -n '__fish_spack_using_command cd' -s e -l env -r -f -a location_env
complete -c spack -n '__fish_spack_using_command cd' -s e -l env -r -d 'location of the named or current environment'
complete -c spack -n '__fish_spack_using_command cd' -l first -f -a find_first
complete -c spack -n '__fish_spack_using_command cd' -l first -d 'use the first match if multiple packages match the spec'

# spack change
set -g __fish_spack_optspecs_spack_change h/help l/list-name= match-spec= a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 change' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command change' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command change' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command change' -s l -l list-name -r -f -a list_name
complete -c spack -n '__fish_spack_using_command change' -s l -l list-name -r -d 'name of the list to remove specs from'
complete -c spack -n '__fish_spack_using_command change' -l match-spec -r -f -a match_spec
complete -c spack -n '__fish_spack_using_command change' -l match-spec -r -d 'if name is ambiguous, supply a spec to match'
complete -c spack -n '__fish_spack_using_command change' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command change' -s a -l all -d 'change all matching specs (allow changing more than one spec)'

# spack checksum
set -g __fish_spack_optspecs_spack_checksum h/help keep-stage b/batch l/latest p/preferred a/add-to-package verify j/jobs=
complete -c spack -n '__fish_spack_using_command_pos 0 checksum' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 checksum' -f -a '(__fish_spack_package_versions $__fish_spack_argparse_argv[1])'
complete -c spack -n '__fish_spack_using_command checksum' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command checksum' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command checksum' -l keep-stage -f -a keep_stage
complete -c spack -n '__fish_spack_using_command checksum' -l keep-stage -d 'don'"'"'t clean up staging area when command completes'
complete -c spack -n '__fish_spack_using_command checksum' -l batch -s b -f -a batch
complete -c spack -n '__fish_spack_using_command checksum' -l batch -s b -d 'don'"'"'t ask which versions to checksum'
complete -c spack -n '__fish_spack_using_command checksum' -l latest -s l -f -a latest
complete -c spack -n '__fish_spack_using_command checksum' -l latest -s l -d 'checksum the latest available version'
complete -c spack -n '__fish_spack_using_command checksum' -l preferred -s p -f -a preferred
complete -c spack -n '__fish_spack_using_command checksum' -l preferred -s p -d 'checksum the known Spack preferred version'
complete -c spack -n '__fish_spack_using_command checksum' -l add-to-package -s a -f -a add_to_package
complete -c spack -n '__fish_spack_using_command checksum' -l add-to-package -s a -d 'add new versions to package'
complete -c spack -n '__fish_spack_using_command checksum' -l verify -f -a verify
complete -c spack -n '__fish_spack_using_command checksum' -l verify -d 'verify known package checksums'
complete -c spack -n '__fish_spack_using_command checksum' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command checksum' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack ci
set -g __fish_spack_optspecs_spack_ci h/help
complete -c spack -n '__fish_spack_using_command_pos 0 ci' -f -a generate -d 'generate jobs file from a CI-aware spack file'
complete -c spack -n '__fish_spack_using_command_pos 0 ci' -f -a rebuild-index -d 'rebuild the buildcache index for the remote mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 ci' -f -a rebuild -d 'rebuild a spec if it is not on the remote mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 ci' -f -a reproduce-build -d 'generate instructions for reproducing the spec rebuild job'
complete -c spack -n '__fish_spack_using_command_pos 0 ci' -f -a verify-versions -d 'validate version checksum & commits between git refs'
complete -c spack -n '__fish_spack_using_command ci' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci' -s h -l help -d 'show this help message and exit'

# spack ci generate
set -g __fish_spack_optspecs_spack_ci_generate h/help output-file= prune-dag no-prune-dag prune-unaffected no-prune-unaffected prune-externals no-prune-externals check-index-only artifacts-root= f/force U/fresh reuse fresh-roots deprecated j/jobs=
complete -c spack -n '__fish_spack_using_command ci generate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci generate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command ci generate' -l output-file -r -f -a output_file
complete -c spack -n '__fish_spack_using_command ci generate' -l output-file -r -d 'pathname for the generated gitlab ci yaml file'
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-dag -f -a prune_dag
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-dag -d 'skip up-to-date specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-dag -f -a prune_dag
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-dag -d 'process up-to-date specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-unaffected -f -a prune_unaffected
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-unaffected -d 'skip up-to-date specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-unaffected -f -a prune_unaffected
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-unaffected -d 'process up-to-date specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-externals -f -a prune_externals
complete -c spack -n '__fish_spack_using_command ci generate' -l prune-externals -d 'skip external specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-externals -f -a prune_externals
complete -c spack -n '__fish_spack_using_command ci generate' -l no-prune-externals -d 'process external specs'
complete -c spack -n '__fish_spack_using_command ci generate' -l check-index-only -f -a index_only
complete -c spack -n '__fish_spack_using_command ci generate' -l check-index-only -d 'only check spec state from buildcache indices'
complete -c spack -n '__fish_spack_using_command ci generate' -l artifacts-root -r -f -a artifacts_root
complete -c spack -n '__fish_spack_using_command ci generate' -l artifacts-root -r -d 'path to the root of the artifacts directory'
complete -c spack -n '__fish_spack_using_command ci generate' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command ci generate' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command ci generate' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command ci generate' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command ci generate' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command ci generate' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command ci generate' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command ci generate' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command ci generate' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command ci generate' -l deprecated -d 'allow concretizer to select deprecated versions'
complete -c spack -n '__fish_spack_using_command ci generate' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command ci generate' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack ci rebuild-index
set -g __fish_spack_optspecs_spack_ci_rebuild_index h/help
complete -c spack -n '__fish_spack_using_command ci rebuild-index' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci rebuild-index' -s h -l help -d 'show this help message and exit'

# spack ci rebuild
set -g __fish_spack_optspecs_spack_ci_rebuild h/help t/tests fail-fast timeout= j/jobs=
complete -c spack -n '__fish_spack_using_command ci rebuild' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci rebuild' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command ci rebuild' -s t -l tests -f -a tests
complete -c spack -n '__fish_spack_using_command ci rebuild' -s t -l tests -d 'run stand-alone tests after the build'
complete -c spack -n '__fish_spack_using_command ci rebuild' -l fail-fast -f -a fail_fast
complete -c spack -n '__fish_spack_using_command ci rebuild' -l fail-fast -d 'stop stand-alone tests after the first failure'
complete -c spack -n '__fish_spack_using_command ci rebuild' -l timeout -r -f -a timeout
complete -c spack -n '__fish_spack_using_command ci rebuild' -l timeout -r -d 'maximum time (in seconds) that tests are allowed to run'
complete -c spack -n '__fish_spack_using_command ci rebuild' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command ci rebuild' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack ci reproduce-build
set -g __fish_spack_optspecs_spack_ci_reproduce_build h/help runtime= working-dir= s/autostart use-local-head gpg-file= gpg-url=
complete -c spack -n '__fish_spack_using_command_pos 0 ci reproduce-build' -f
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l runtime -r -f -a 'docker podman'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l runtime -r -d 'Container runtime to use.'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l working-dir -r -f -a working_dir
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l working-dir -r -d 'where to unpack artifacts'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -s s -l autostart -f -a autostart
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -s s -l autostart -d 'Run docker reproducer automatically'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l use-local-head -f -a use_local_head
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l use-local-head -d 'Use the HEAD of the local Spack instead of reproducing a commit'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l gpg-file -r -f -a gpg_file
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l gpg-file -r -d 'Path to public GPG key for validating binary cache installs'
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l gpg-url -r -f -a gpg_url
complete -c spack -n '__fish_spack_using_command ci reproduce-build' -l gpg-url -r -d 'URL to public GPG key for validating binary cache installs'

# spack ci verify-versions
set -g __fish_spack_optspecs_spack_ci_verify_versions h/help

complete -c spack -n '__fish_spack_using_command ci verify-versions' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command ci verify-versions' -s h -l help -d 'show this help message and exit'

# spack clean
set -g __fish_spack_optspecs_spack_clean h/help s/stage d/downloads f/failures m/misc-cache p/python-cache b/bootstrap a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 clean' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command clean' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command clean' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command clean' -s s -l stage -f -a stage
complete -c spack -n '__fish_spack_using_command clean' -s s -l stage -d 'remove all temporary build stages (default)'
complete -c spack -n '__fish_spack_using_command clean' -s d -l downloads -f -a downloads
complete -c spack -n '__fish_spack_using_command clean' -s d -l downloads -d 'remove cached downloads'
complete -c spack -n '__fish_spack_using_command clean' -s f -l failures -f -a failures
complete -c spack -n '__fish_spack_using_command clean' -s f -l failures -d 'force removal of all install failure tracking markers'
complete -c spack -n '__fish_spack_using_command clean' -s m -l misc-cache -f -a misc_cache
complete -c spack -n '__fish_spack_using_command clean' -s m -l misc-cache -d 'remove long-lived caches, like the virtual package index'
complete -c spack -n '__fish_spack_using_command clean' -s p -l python-cache -f -a python_cache
complete -c spack -n '__fish_spack_using_command clean' -s p -l python-cache -d 'remove .pyc, .pyo files and __pycache__ folders'
complete -c spack -n '__fish_spack_using_command clean' -s b -l bootstrap -f -a bootstrap
complete -c spack -n '__fish_spack_using_command clean' -s b -l bootstrap -d 'remove software and configuration needed to bootstrap Spack'
complete -c spack -n '__fish_spack_using_command clean' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command clean' -s a -l all -d 'equivalent to -sdfmp (does not include --bootstrap)'

# spack commands
set -g __fish_spack_optspecs_spack_commands h/help update-completion a/aliases format= header= update=

complete -c spack -n '__fish_spack_using_command commands' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command commands' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command commands' -l update-completion -f -a update_completion
complete -c spack -n '__fish_spack_using_command commands' -l update-completion -d 'regenerate spack'"'"'s tab completion scripts'
complete -c spack -n '__fish_spack_using_command commands' -s a -l aliases -f -a aliases
complete -c spack -n '__fish_spack_using_command commands' -s a -l aliases -d 'include command aliases'
complete -c spack -n '__fish_spack_using_command commands' -l format -r -f -a 'subcommands rst names bash fish'
complete -c spack -n '__fish_spack_using_command commands' -l format -r -d 'format to be used to print the output (default: names)'
complete -c spack -n '__fish_spack_using_command commands' -l header -r -f -a header
complete -c spack -n '__fish_spack_using_command commands' -l header -r -d 'prepend contents of FILE to the output (useful for rst format)'
complete -c spack -n '__fish_spack_using_command commands' -l update -r -f -a update
complete -c spack -n '__fish_spack_using_command commands' -l update -r -d 'write output to the specified file, if any command is newer'

# spack compiler
set -g __fish_spack_optspecs_spack_compiler h/help
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a find -d 'search the system for compilers to add to Spack configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a add -d 'search the system for compilers to add to Spack configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a remove -d 'remove compiler by spec'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a rm -d 'remove compiler by spec'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a list -d 'list available compilers'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a ls -d 'list available compilers'
complete -c spack -n '__fish_spack_using_command_pos 0 compiler' -f -a info -d 'show compiler paths'
complete -c spack -n '__fish_spack_using_command compiler' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler' -s h -l help -d 'show this help message and exit'

# spack compiler find
set -g __fish_spack_optspecs_spack_compiler_find h/help mixed-toolchain no-mixed-toolchain scope= j/jobs=

complete -c spack -n '__fish_spack_using_command compiler find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler find' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler find' -l mixed-toolchain -f -a mixed_toolchain
complete -c spack -n '__fish_spack_using_command compiler find' -l mixed-toolchain -d '(DEPRECATED) Allow mixed toolchains (for example: clang, clang++, gfortran)'
complete -c spack -n '__fish_spack_using_command compiler find' -l no-mixed-toolchain -f -a mixed_toolchain
complete -c spack -n '__fish_spack_using_command compiler find' -l no-mixed-toolchain -d '(DEPRECATED) Do not allow mixed toolchains (for example: clang, clang++, gfortran)'
complete -c spack -n '__fish_spack_using_command compiler find' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler find' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command compiler find' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command compiler find' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack compiler add
set -g __fish_spack_optspecs_spack_compiler_add h/help mixed-toolchain no-mixed-toolchain scope= j/jobs=

complete -c spack -n '__fish_spack_using_command compiler add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler add' -l mixed-toolchain -f -a mixed_toolchain
complete -c spack -n '__fish_spack_using_command compiler add' -l mixed-toolchain -d '(DEPRECATED) Allow mixed toolchains (for example: clang, clang++, gfortran)'
complete -c spack -n '__fish_spack_using_command compiler add' -l no-mixed-toolchain -f -a mixed_toolchain
complete -c spack -n '__fish_spack_using_command compiler add' -l no-mixed-toolchain -d '(DEPRECATED) Do not allow mixed toolchains (for example: clang, clang++, gfortran)'
complete -c spack -n '__fish_spack_using_command compiler add' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler add' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command compiler add' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command compiler add' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack compiler remove
set -g __fish_spack_optspecs_spack_compiler_remove h/help a/all scope=
complete -c spack -n '__fish_spack_using_command_pos 0 compiler remove' -f -a '(__fish_spack_installed_compilers)'
complete -c spack -n '__fish_spack_using_command compiler remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler remove' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command compiler remove' -s a -l all -d 'remove ALL compilers that match spec'
complete -c spack -n '__fish_spack_using_command compiler remove' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler remove' -l scope -r -d 'configuration scope to modify'

# spack compiler rm
set -g __fish_spack_optspecs_spack_compiler_rm h/help a/all scope=
complete -c spack -n '__fish_spack_using_command_pos 0 compiler rm' -f -a '(__fish_spack_installed_compilers)'
complete -c spack -n '__fish_spack_using_command compiler rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler rm' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command compiler rm' -s a -l all -d 'remove ALL compilers that match spec'
complete -c spack -n '__fish_spack_using_command compiler rm' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler rm' -l scope -r -d 'configuration scope to modify'

# spack compiler list
set -g __fish_spack_optspecs_spack_compiler_list h/help scope= remote
complete -c spack -n '__fish_spack_using_command compiler list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler list' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler list' -l scope -r -d 'configuration scope to read from'
complete -c spack -n '__fish_spack_using_command compiler list' -l remote -f -a remote
complete -c spack -n '__fish_spack_using_command compiler list' -l remote -d 'list also compilers from registered buildcaches'

# spack compiler ls
set -g __fish_spack_optspecs_spack_compiler_ls h/help scope= remote
complete -c spack -n '__fish_spack_using_command compiler ls' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler ls' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler ls' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler ls' -l scope -r -d 'configuration scope to read from'
complete -c spack -n '__fish_spack_using_command compiler ls' -l remote -f -a remote
complete -c spack -n '__fish_spack_using_command compiler ls' -l remote -d 'list also compilers from registered buildcaches'

# spack compiler info
set -g __fish_spack_optspecs_spack_compiler_info h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 compiler info' -f -a '(__fish_spack_installed_compilers)'
complete -c spack -n '__fish_spack_using_command compiler info' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compiler info' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compiler info' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compiler info' -l scope -r -d 'configuration scope to read from'

# spack compilers
set -g __fish_spack_optspecs_spack_compilers h/help scope= remote
complete -c spack -n '__fish_spack_using_command compilers' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command compilers' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command compilers' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command compilers' -l scope -r -d 'configuration scope to read/modify'
complete -c spack -n '__fish_spack_using_command compilers' -l remote -f -a remote
complete -c spack -n '__fish_spack_using_command compilers' -l remote -d 'list also compilers from registered buildcaches'

# spack concretize
set -g __fish_spack_optspecs_spack_concretize h/help test= q/quiet f/force U/fresh reuse fresh-roots deprecated j/jobs=
complete -c spack -n '__fish_spack_using_command concretize' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command concretize' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command concretize' -l test -r -f -a 'root all'
complete -c spack -n '__fish_spack_using_command concretize' -l test -r -d 'concretize with test dependencies of only root packages or all packages'
complete -c spack -n '__fish_spack_using_command concretize' -s q -l quiet -f -a quiet
complete -c spack -n '__fish_spack_using_command concretize' -s q -l quiet -d 'don'"'"'t print concretized specs'
complete -c spack -n '__fish_spack_using_command concretize' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command concretize' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command concretize' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretize' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command concretize' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretize' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command concretize' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretize' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command concretize' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command concretize' -l deprecated -d 'allow concretizer to select deprecated versions'
complete -c spack -n '__fish_spack_using_command concretize' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command concretize' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack concretise
set -g __fish_spack_optspecs_spack_concretise h/help test= q/quiet f/force U/fresh reuse fresh-roots deprecated j/jobs=
complete -c spack -n '__fish_spack_using_command concretise' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command concretise' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command concretise' -l test -r -f -a 'root all'
complete -c spack -n '__fish_spack_using_command concretise' -l test -r -d 'concretize with test dependencies of only root packages or all packages'
complete -c spack -n '__fish_spack_using_command concretise' -s q -l quiet -f -a quiet
complete -c spack -n '__fish_spack_using_command concretise' -s q -l quiet -d 'don'"'"'t print concretized specs'
complete -c spack -n '__fish_spack_using_command concretise' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command concretise' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command concretise' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretise' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command concretise' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretise' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command concretise' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command concretise' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command concretise' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command concretise' -l deprecated -d 'allow concretizer to select deprecated versions'
complete -c spack -n '__fish_spack_using_command concretise' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command concretise' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack config
set -g __fish_spack_optspecs_spack_config h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a get -d 'print configuration values'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a blame -d 'print configuration annotated with source file:line'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a edit -d 'edit configuration file'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a list -d 'list configuration sections'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a scopes -d 'list defined scopes in descending order of precedence'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a add -d 'add configuration parameters'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a change -d 'swap variants etc. on specs in config'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a prefer-upstream -d 'set package preferences from upstream'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a remove -d 'remove configuration parameters'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a rm -d 'remove configuration parameters'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a update -d 'update configuration files to the latest format'
complete -c spack -n '__fish_spack_using_command_pos 0 config' -f -a revert -d 'revert configuration files to their state before update'
complete -c spack -n '__fish_spack_using_command config' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command config' -l scope -r -d 'configuration scope to read/modify'

# spack config get
set -g __fish_spack_optspecs_spack_config_get h/help
complete -c spack -n '__fish_spack_using_command_pos 0 config get' -f -a 'bootstrap cdash ci compilers concretizer config definitions develop env_vars include mirrors modules packages repos toolchains upstreams view'
complete -c spack -n '__fish_spack_using_command config get' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config get' -s h -l help -d 'show this help message and exit'

# spack config blame
set -g __fish_spack_optspecs_spack_config_blame h/help
complete -c spack -n '__fish_spack_using_command_pos 0 config blame' -f -a 'bootstrap cdash ci compilers concretizer config definitions develop env_vars include mirrors modules packages repos toolchains upstreams view'
complete -c spack -n '__fish_spack_using_command config blame' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config blame' -s h -l help -d 'show this help message and exit'

# spack config edit
set -g __fish_spack_optspecs_spack_config_edit h/help print-file
complete -c spack -n '__fish_spack_using_command_pos 0 config edit' -f -a 'bootstrap cdash ci compilers concretizer config definitions develop env_vars include mirrors modules packages repos toolchains upstreams view'
complete -c spack -n '__fish_spack_using_command config edit' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config edit' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config edit' -l print-file -f -a print_file
complete -c spack -n '__fish_spack_using_command config edit' -l print-file -d 'print the file name that would be edited'

# spack config list
set -g __fish_spack_optspecs_spack_config_list h/help
complete -c spack -n '__fish_spack_using_command config list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config list' -s h -l help -d 'show this help message and exit'

# spack config scopes
set -g __fish_spack_optspecs_spack_config_scopes h/help p/paths t/type=
complete -c spack -n '__fish_spack_using_command_pos 0 config scopes' -f -a 'bootstrap cdash ci compilers concretizer config definitions develop env_vars include mirrors modules packages repos toolchains upstreams view'
complete -c spack -n '__fish_spack_using_command config scopes' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config scopes' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config scopes' -s p -l paths -f -a paths
complete -c spack -n '__fish_spack_using_command config scopes' -s p -l paths -d 'show associated paths for appropriate scopes'
complete -c spack -n '__fish_spack_using_command config scopes' -s t -l type -r -f -a 'all env include internal path'
complete -c spack -n '__fish_spack_using_command config scopes' -s t -l type -r -d 'list only scopes of the specified type(s)'

# spack config add
set -g __fish_spack_optspecs_spack_config_add h/help f/file=
complete -c spack -n '__fish_spack_using_command_pos 0 config add' -f -a '(__fish_spack_colon_path)'
complete -c spack -n '__fish_spack_using_command config add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config add' -s f -l file -r -f -a file
complete -c spack -n '__fish_spack_using_command config add' -s f -l file -r -d 'file from which to set all config values'

# spack config change
set -g __fish_spack_optspecs_spack_config_change h/help match-spec=
complete -c spack -n '__fish_spack_using_command_pos 0 config change' -f -a '(__fish_spack_colon_path)'
complete -c spack -n '__fish_spack_using_command config change' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config change' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config change' -l match-spec -r -f -a match_spec
complete -c spack -n '__fish_spack_using_command config change' -l match-spec -r -d 'only change constraints that match this'

# spack config prefer-upstream
set -g __fish_spack_optspecs_spack_config_prefer_upstream h/help local
complete -c spack -n '__fish_spack_using_command config prefer-upstream' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config prefer-upstream' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config prefer-upstream' -l local -f -a local
complete -c spack -n '__fish_spack_using_command config prefer-upstream' -l local -d 'set packages preferences based on local installs, rather than upstream'

# spack config remove
set -g __fish_spack_optspecs_spack_config_remove h/help
complete -c spack -n '__fish_spack_using_command_pos 0 config remove' -f -a '(__fish_spack_colon_path)'
complete -c spack -n '__fish_spack_using_command config remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config remove' -s h -l help -d 'show this help message and exit'

# spack config rm
set -g __fish_spack_optspecs_spack_config_rm h/help
complete -c spack -n '__fish_spack_using_command_pos 0 config rm' -f -a '(__fish_spack_colon_path)'
complete -c spack -n '__fish_spack_using_command config rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config rm' -s h -l help -d 'show this help message and exit'

# spack config update
set -g __fish_spack_optspecs_spack_config_update h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos 0 config update' -f -a '(__fish_spack_config_sections)'
complete -c spack -n '__fish_spack_using_command config update' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config update' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config update' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command config update' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack config revert
set -g __fish_spack_optspecs_spack_config_revert h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos 0 config revert' -f -a '(__fish_spack_config_sections)'
complete -c spack -n '__fish_spack_using_command config revert' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command config revert' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command config revert' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command config revert' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack containerize
set -g __fish_spack_optspecs_spack_containerize h/help list-os last-stage=
complete -c spack -n '__fish_spack_using_command containerize' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command containerize' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command containerize' -l list-os -f -a list_os
complete -c spack -n '__fish_spack_using_command containerize' -l list-os -d 'list all the OS that can be used in the bootstrap phase and exit'
complete -c spack -n '__fish_spack_using_command containerize' -l last-stage -r -f -a 'bootstrap build final'
complete -c spack -n '__fish_spack_using_command containerize' -l last-stage -r -d 'last stage in the container recipe'

# spack containerise
set -g __fish_spack_optspecs_spack_containerise h/help list-os last-stage=
complete -c spack -n '__fish_spack_using_command containerise' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command containerise' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command containerise' -l list-os -f -a list_os
complete -c spack -n '__fish_spack_using_command containerise' -l list-os -d 'list all the OS that can be used in the bootstrap phase and exit'
complete -c spack -n '__fish_spack_using_command containerise' -l last-stage -r -f -a 'bootstrap build final'
complete -c spack -n '__fish_spack_using_command containerise' -l last-stage -r -d 'last stage in the container recipe'

# spack create
set -g __fish_spack_optspecs_spack_create h/help keep-stage n/name= t/template= r/repo= N/namespace= f/force skip-editor b/batch

complete -c spack -n '__fish_spack_using_command create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command create' -l keep-stage -f -a keep_stage
complete -c spack -n '__fish_spack_using_command create' -l keep-stage -d 'don'"'"'t clean up staging area when command completes'
complete -c spack -n '__fish_spack_using_command create' -s n -l name -r -f -a name
complete -c spack -n '__fish_spack_using_command create' -s n -l name -r -d 'name of the package to create'
complete -c spack -n '__fish_spack_using_command create' -s t -l template -r -f -a 'autoreconf autotools bazel bundle cargo cmake generic go intel lua makefile maven meson octave perlbuild perlmake python qmake r racket ruby scons sip waf'
complete -c spack -n '__fish_spack_using_command create' -s t -l template -r -d 'build system template to use'
complete -c spack -n '__fish_spack_using_command create' -s r -l repo -r -f -a repo
complete -c spack -n '__fish_spack_using_command create' -s r -l repo -r -d 'path to a repository where the package should be created'
complete -c spack -n '__fish_spack_using_command create' -s N -l namespace -r -f -a namespace
complete -c spack -n '__fish_spack_using_command create' -s N -l namespace -r -d 'specify a namespace for the package'
complete -c spack -n '__fish_spack_using_command create' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command create' -s f -l force -d 'overwrite any existing package file with the same name'
complete -c spack -n '__fish_spack_using_command create' -l skip-editor -f -a skip_editor
complete -c spack -n '__fish_spack_using_command create' -l skip-editor -d 'skip the edit session for the package (e.g., automation)'
complete -c spack -n '__fish_spack_using_command create' -s b -l batch -f -a batch
complete -c spack -n '__fish_spack_using_command create' -s b -l batch -d 'don'"'"'t ask which versions to checksum'

# spack debug
set -g __fish_spack_optspecs_spack_debug h/help
complete -c spack -n '__fish_spack_using_command_pos 0 debug' -f -a report -d 'print information useful for bug reports'
complete -c spack -n '__fish_spack_using_command debug' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command debug' -s h -l help -d 'show this help message and exit'

# spack debug report
set -g __fish_spack_optspecs_spack_debug_report h/help
complete -c spack -n '__fish_spack_using_command debug report' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command debug report' -s h -l help -d 'show this help message and exit'

# spack deconcretize
set -g __fish_spack_optspecs_spack_deconcretize h/help root y/yes-to-all a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 deconcretize' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command deconcretize' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command deconcretize' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command deconcretize' -l root -f -a root
complete -c spack -n '__fish_spack_using_command deconcretize' -l root -d 'deconcretize only specific environment roots'
complete -c spack -n '__fish_spack_using_command deconcretize' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command deconcretize' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command deconcretize' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command deconcretize' -s a -l all -d 'deconcretize ALL specs that match each supplied spec'

# spack dependencies
set -g __fish_spack_optspecs_spack_dependencies h/help i/installed t/transitive deptype= V/no-expand-virtuals
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 dependencies' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command dependencies' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command dependencies' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command dependencies' -s i -l installed -f -a installed
complete -c spack -n '__fish_spack_using_command dependencies' -s i -l installed -d 'list installed dependencies of an installed spec instead of possible dependencies of a package'
complete -c spack -n '__fish_spack_using_command dependencies' -s t -l transitive -f -a transitive
complete -c spack -n '__fish_spack_using_command dependencies' -s t -l transitive -d 'show all transitive dependencies'
complete -c spack -n '__fish_spack_using_command dependencies' -l deptype -r -f -a deptype
complete -c spack -n '__fish_spack_using_command dependencies' -l deptype -r -d 'comma-separated list of deptypes to traverse (default=build,link,run,test)'
complete -c spack -n '__fish_spack_using_command dependencies' -s V -l no-expand-virtuals -f -a expand_virtuals
complete -c spack -n '__fish_spack_using_command dependencies' -s V -l no-expand-virtuals -d 'do not expand virtual dependencies'

# spack dependents
set -g __fish_spack_optspecs_spack_dependents h/help i/installed t/transitive
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 dependents' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command dependents' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command dependents' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command dependents' -s i -l installed -f -a installed
complete -c spack -n '__fish_spack_using_command dependents' -s i -l installed -d 'list installed dependents of an installed spec instead of possible dependents of a package'
complete -c spack -n '__fish_spack_using_command dependents' -s t -l transitive -f -a transitive
complete -c spack -n '__fish_spack_using_command dependents' -s t -l transitive -d 'show all transitive dependents'

# spack deprecate
set -g __fish_spack_optspecs_spack_deprecate h/help y/yes-to-all d/dependencies D/no-dependencies i/install-deprecator I/no-install-deprecator l/link-type=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 deprecate' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command deprecate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command deprecate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command deprecate' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command deprecate' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command deprecate' -s d -l dependencies -f -a dependencies
complete -c spack -n '__fish_spack_using_command deprecate' -s d -l dependencies -d 'deprecate dependencies (default)'
complete -c spack -n '__fish_spack_using_command deprecate' -s D -l no-dependencies -f -a dependencies
complete -c spack -n '__fish_spack_using_command deprecate' -s D -l no-dependencies -d 'do not deprecate dependencies'
complete -c spack -n '__fish_spack_using_command deprecate' -s i -l install-deprecator -f -a install
complete -c spack -n '__fish_spack_using_command deprecate' -s i -l install-deprecator -d 'concretize and install deprecator spec'
complete -c spack -n '__fish_spack_using_command deprecate' -s I -l no-install-deprecator -f -a install
complete -c spack -n '__fish_spack_using_command deprecate' -s I -l no-install-deprecator -d 'deprecator spec must already be installed (default)'
complete -c spack -n '__fish_spack_using_command deprecate' -s l -l link-type -r -f -a 'soft hard'
complete -c spack -n '__fish_spack_using_command deprecate' -s l -l link-type -r -d '(deprecated)'

# spack dev-build
set -g __fish_spack_optspecs_spack_dev_build h/help j/jobs= n/no-checksum d/source-path= i/ignore-dependencies keep-prefix skip-patch q/quiet drop-in= test= b/before= u/until= clean dirty f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 dev-build' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command dev-build' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command dev-build' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command dev-build' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command dev-build' -s j -l jobs -r -d 'explicitly set number of parallel jobs'
complete -c spack -n '__fish_spack_using_command dev-build' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command dev-build' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'
complete -c spack -n '__fish_spack_using_command dev-build' -s d -l source-path -r -f -a source_path
complete -c spack -n '__fish_spack_using_command dev-build' -s d -l source-path -r -d 'path to source directory (defaults to the current directory)'
complete -c spack -n '__fish_spack_using_command dev-build' -s i -l ignore-dependencies -f -a ignore_deps
complete -c spack -n '__fish_spack_using_command dev-build' -s i -l ignore-dependencies -d 'do not try to install dependencies of requested packages'
complete -c spack -n '__fish_spack_using_command dev-build' -l keep-prefix -f -a keep_prefix
complete -c spack -n '__fish_spack_using_command dev-build' -l keep-prefix -d 'do not remove the install prefix if installation fails'
complete -c spack -n '__fish_spack_using_command dev-build' -l skip-patch -f -a skip_patch
complete -c spack -n '__fish_spack_using_command dev-build' -l skip-patch -d 'skip patching for the developer build'
complete -c spack -n '__fish_spack_using_command dev-build' -s q -l quiet -f -a quiet
complete -c spack -n '__fish_spack_using_command dev-build' -s q -l quiet -d 'do not display verbose build output while installing'
complete -c spack -n '__fish_spack_using_command dev-build' -l drop-in -r -f -a shell
complete -c spack -n '__fish_spack_using_command dev-build' -l drop-in -r -d 'drop into a build environment in a new shell, e.g., bash'
complete -c spack -n '__fish_spack_using_command dev-build' -l test -r -f -a 'root all'
complete -c spack -n '__fish_spack_using_command dev-build' -l test -r -d 'run tests on only root packages or all packages'
complete -c spack -n '__fish_spack_using_command dev-build' -s b -l before -r -f -a before
complete -c spack -n '__fish_spack_using_command dev-build' -s b -l before -r -d 'phase to stop before when installing (default None)'
complete -c spack -n '__fish_spack_using_command dev-build' -s u -l until -r -f -a until
complete -c spack -n '__fish_spack_using_command dev-build' -s u -l until -r -d 'phase to stop after when installing (default None)'
complete -c spack -n '__fish_spack_using_command dev-build' -l clean -f -a dirty
complete -c spack -n '__fish_spack_using_command dev-build' -l clean -d 'unset harmful variables in the build environment (default)'
complete -c spack -n '__fish_spack_using_command dev-build' -l dirty -f -a dirty
complete -c spack -n '__fish_spack_using_command dev-build' -l dirty -d 'preserve user environment in spack'"'"'s build environment (danger!)'
complete -c spack -n '__fish_spack_using_command dev-build' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command dev-build' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command dev-build' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command dev-build' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command dev-build' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command dev-build' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command dev-build' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command dev-build' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command dev-build' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command dev-build' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack develop
set -g __fish_spack_optspecs_spack_develop h/help p/path= b/build-directory= no-clone clone f/force= r/recursive
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 develop' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command develop' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command develop' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command develop' -s p -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command develop' -s p -l path -r -d 'source location of package'
complete -c spack -n '__fish_spack_using_command develop' -s b -l build-directory -r -f -a build_directory
complete -c spack -n '__fish_spack_using_command develop' -s b -l build-directory -r -d 'build directory for the package'
complete -c spack -n '__fish_spack_using_command develop' -l no-clone -f -a clone
complete -c spack -n '__fish_spack_using_command develop' -l no-clone -d 'do not clone, the package already exists at the source path'
complete -c spack -n '__fish_spack_using_command develop' -l clone -f -a clone
complete -c spack -n '__fish_spack_using_command develop' -l clone -d '(default) clone the package unless the path already exists, use --force to overwrite'
complete -c spack -n '__fish_spack_using_command develop' -s f -l force -r -f -a force
complete -c spack -n '__fish_spack_using_command develop' -s f -l force -r -d 'remove any files or directories that block cloning source code'
complete -c spack -n '__fish_spack_using_command develop' -s r -l recursive -f -a recursive
complete -c spack -n '__fish_spack_using_command develop' -s r -l recursive -d 'traverse nodes of the graph to mark everything up to the root as a develop spec'

# spack diff
set -g __fish_spack_optspecs_spack_diff h/help json first a/attribute= ignore=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 diff' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command diff' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command diff' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command diff' -l json -f -a dump_json
complete -c spack -n '__fish_spack_using_command diff' -l json -d 'dump json output instead of pretty printing'
complete -c spack -n '__fish_spack_using_command diff' -l first -f -a load_first
complete -c spack -n '__fish_spack_using_command diff' -l first -d 'load the first match if multiple packages match the spec'
complete -c spack -n '__fish_spack_using_command diff' -s a -l attribute -r -f -a attribute
complete -c spack -n '__fish_spack_using_command diff' -s a -l attribute -r -d 'select the attributes to show (defaults to all)'
complete -c spack -n '__fish_spack_using_command diff' -l ignore -r -f -a ignore
complete -c spack -n '__fish_spack_using_command diff' -l ignore -r -d 'omit diffs related to these dependencies'

# spack docs
set -g __fish_spack_optspecs_spack_docs h/help
complete -c spack -n '__fish_spack_using_command docs' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command docs' -s h -l help -d 'show this help message and exit'

# spack edit
set -g __fish_spack_optspecs_spack_edit h/help b/build-system c/command d/docs t/test m/module r/repo= N/namespace=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 edit' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command edit' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command edit' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command edit' -s b -l build-system -f -a path
complete -c spack -n '__fish_spack_using_command edit' -s b -l build-system -d 'edit the build system with the supplied name or fullname'
complete -c spack -n '__fish_spack_using_command edit' -s c -l command -f -a path
complete -c spack -n '__fish_spack_using_command edit' -s c -l command -d 'edit the command with the supplied name'
complete -c spack -n '__fish_spack_using_command edit' -s d -l docs -f -a path
complete -c spack -n '__fish_spack_using_command edit' -s d -l docs -d 'edit the docs with the supplied name'
complete -c spack -n '__fish_spack_using_command edit' -s t -l test -f -a path
complete -c spack -n '__fish_spack_using_command edit' -s t -l test -d 'edit the test with the supplied name'
complete -c spack -n '__fish_spack_using_command edit' -s m -l module -f -a path
complete -c spack -n '__fish_spack_using_command edit' -s m -l module -d 'edit the main spack module with the supplied name'
complete -c spack -n '__fish_spack_using_command edit' -s r -l repo -r -f -a repo
complete -c spack -n '__fish_spack_using_command edit' -s r -l repo -r -d 'path to repo to edit package or build system in'
complete -c spack -n '__fish_spack_using_command edit' -s N -l namespace -r -f -a namespace
complete -c spack -n '__fish_spack_using_command edit' -s N -l namespace -r -d 'namespace of package or build system to edit'

# spack env
set -g __fish_spack_optspecs_spack_env h/help
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a activate -d 'set the active environment'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a deactivate -d 'deactivate the active environment'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a create -d 'create a new environment'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a remove -d 'remove managed environment(s)'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a rm -d 'remove managed environment(s)'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a rename -d 'rename an existing environment'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a mv -d 'rename an existing environment'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a list -d 'list all managed environments'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a ls -d 'list all managed environments'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a status -d 'print active environment status'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a st -d 'print active environment status'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a loads -d 'list modules for an installed environment '"'"'(see spack module loads)'"'"''
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a view -d 'manage the environment'"'"'s view'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a update -d 'update the environment manifest to the latest schema format'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a revert -d 'restore the environment manifest to its previous format'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a depfile -d 'generate a depfile to exploit parallel builds across specs'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a track -d 'track an environment from a directory in Spack'
complete -c spack -n '__fish_spack_using_command_pos 0 env' -f -a untrack -d 'track an environment from a directory in Spack'
complete -c spack -n '__fish_spack_using_command env' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env' -s h -l help -d 'show this help message and exit'

# spack env activate
set -g __fish_spack_optspecs_spack_env_activate h/help sh csh fish bat pwsh v/with-view= V/without-view p/prompt temp create envfile= keep-relative d/dir
complete -c spack -n '__fish_spack_using_command_pos 0 env activate' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env activate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env activate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env activate' -l sh -f -a shell
complete -c spack -n '__fish_spack_using_command env activate' -l sh -d 'print sh commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env activate' -l csh -f -a shell
complete -c spack -n '__fish_spack_using_command env activate' -l csh -d 'print csh commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env activate' -l fish -f -a shell
complete -c spack -n '__fish_spack_using_command env activate' -l fish -d 'print fish commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env activate' -l bat -f -a shell
complete -c spack -n '__fish_spack_using_command env activate' -l bat -d 'print bat commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env activate' -l pwsh -f -a shell
complete -c spack -n '__fish_spack_using_command env activate' -l pwsh -d 'print powershell commands to activate environment'
complete -c spack -n '__fish_spack_using_command env activate' -s v -l with-view -r -f -a with_view
complete -c spack -n '__fish_spack_using_command env activate' -s v -l with-view -r -d 'set runtime environment variables for the named view'
complete -c spack -n '__fish_spack_using_command env activate' -s V -l without-view -f -a without_view
complete -c spack -n '__fish_spack_using_command env activate' -s V -l without-view -d 'do not set runtime environment variables for any view'
complete -c spack -n '__fish_spack_using_command env activate' -s p -l prompt -f -a prompt
complete -c spack -n '__fish_spack_using_command env activate' -s p -l prompt -d 'add the active environment to the command line prompt'
complete -c spack -n '__fish_spack_using_command env activate' -l temp -f -a temp
complete -c spack -n '__fish_spack_using_command env activate' -l temp -d 'create and activate in a temporary directory'
complete -c spack -n '__fish_spack_using_command env activate' -l create -f -a create
complete -c spack -n '__fish_spack_using_command env activate' -l create -d 'create and activate the environment if it doesn'"'"'t exist'
complete -c spack -n '__fish_spack_using_command env activate' -l envfile -r -f -a envfile
complete -c spack -n '__fish_spack_using_command env activate' -l envfile -r -d 'manifest or lock file (ends with '"'"'.json'"'"' or '"'"'.lock'"'"')'
complete -c spack -n '__fish_spack_using_command env activate' -l keep-relative -f -a keep_relative
complete -c spack -n '__fish_spack_using_command env activate' -l keep-relative -d 'copy envfile'"'"'s relative develop paths verbatim when create'
complete -c spack -n '__fish_spack_using_command env activate' -s d -l dir -f -a dir
complete -c spack -n '__fish_spack_using_command env activate' -s d -l dir -d 'activate environment based on the directory supplied'

# spack env deactivate
set -g __fish_spack_optspecs_spack_env_deactivate h/help sh csh fish bat pwsh
complete -c spack -n '__fish_spack_using_command env deactivate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env deactivate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env deactivate' -l sh -f -a shell
complete -c spack -n '__fish_spack_using_command env deactivate' -l sh -d 'print sh commands to deactivate the environment'
complete -c spack -n '__fish_spack_using_command env deactivate' -l csh -f -a shell
complete -c spack -n '__fish_spack_using_command env deactivate' -l csh -d 'print csh commands to deactivate the environment'
complete -c spack -n '__fish_spack_using_command env deactivate' -l fish -f -a shell
complete -c spack -n '__fish_spack_using_command env deactivate' -l fish -d 'print fish commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env deactivate' -l bat -f -a shell
complete -c spack -n '__fish_spack_using_command env deactivate' -l bat -d 'print bat commands to activate the environment'
complete -c spack -n '__fish_spack_using_command env deactivate' -l pwsh -f -a shell
complete -c spack -n '__fish_spack_using_command env deactivate' -l pwsh -d 'print pwsh commands to activate the environment'

# spack env create
set -g __fish_spack_optspecs_spack_env_create h/help d/dir keep-relative without-view with-view= include-concrete=
complete -c spack -n '__fish_spack_using_command_pos 0 env create' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env create' -s d -l dir -f -a dir
complete -c spack -n '__fish_spack_using_command env create' -s d -l dir -d 'create an environment in a specific directory'
complete -c spack -n '__fish_spack_using_command env create' -l keep-relative -f -a keep_relative
complete -c spack -n '__fish_spack_using_command env create' -l keep-relative -d 'copy envfile'"'"'s relative develop paths verbatim'
complete -c spack -n '__fish_spack_using_command env create' -l without-view -f -a without_view
complete -c spack -n '__fish_spack_using_command env create' -l without-view -d 'do not maintain a view for this environment'
complete -c spack -n '__fish_spack_using_command env create' -l with-view -r -f -a with_view
complete -c spack -n '__fish_spack_using_command env create' -l with-view -r -d 'maintain view at WITH_VIEW (vs. environment'"'"'s directory)'
complete -c spack -n '__fish_spack_using_command env create' -l include-concrete -r -f -a include_concrete
complete -c spack -n '__fish_spack_using_command env create' -l include-concrete -r -d 'copy concrete specs from INCLUDE_CONCRETE'"'"'s environment'

# spack env remove
set -g __fish_spack_optspecs_spack_env_remove h/help y/yes-to-all f/force
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 env remove' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env remove' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env remove' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command env remove' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command env remove' -s f -l force -d 'force removal even when included in other environment(s)'

# spack env rm
set -g __fish_spack_optspecs_spack_env_rm h/help y/yes-to-all f/force
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 env rm' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env rm' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env rm' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command env rm' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command env rm' -s f -l force -d 'force removal even when included in other environment(s)'

# spack env rename
set -g __fish_spack_optspecs_spack_env_rename h/help d/dir f/force

complete -c spack -n '__fish_spack_using_command env rename' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env rename' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env rename' -s d -l dir -f -a dir
complete -c spack -n '__fish_spack_using_command env rename' -s d -l dir -d 'positional arguments are environment directory paths'
complete -c spack -n '__fish_spack_using_command env rename' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command env rename' -s f -l force -d 'force renaming even if overwriting an existing environment'

# spack env mv
set -g __fish_spack_optspecs_spack_env_mv h/help d/dir f/force

complete -c spack -n '__fish_spack_using_command env mv' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env mv' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env mv' -s d -l dir -f -a dir
complete -c spack -n '__fish_spack_using_command env mv' -s d -l dir -d 'positional arguments are environment directory paths'
complete -c spack -n '__fish_spack_using_command env mv' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command env mv' -s f -l force -d 'force renaming even if overwriting an existing environment'

# spack env list
set -g __fish_spack_optspecs_spack_env_list h/help
complete -c spack -n '__fish_spack_using_command env list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env list' -s h -l help -d 'show this help message and exit'

# spack env ls
set -g __fish_spack_optspecs_spack_env_ls h/help
complete -c spack -n '__fish_spack_using_command env ls' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env ls' -s h -l help -d 'show this help message and exit'

# spack env status
set -g __fish_spack_optspecs_spack_env_status h/help
complete -c spack -n '__fish_spack_using_command env status' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env status' -s h -l help -d 'show this help message and exit'

# spack env st
set -g __fish_spack_optspecs_spack_env_st h/help
complete -c spack -n '__fish_spack_using_command env st' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env st' -s h -l help -d 'show this help message and exit'

# spack env loads
set -g __fish_spack_optspecs_spack_env_loads h/help n/module-set-name= m/module-type= input-only p/prefix= x/exclude= r/dependencies
complete -c spack -n '__fish_spack_using_command env loads' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env loads' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env loads' -s n -l module-set-name -r -f -a module_set_name
complete -c spack -n '__fish_spack_using_command env loads' -s n -l module-set-name -r -d 'module set for which to generate load operations'
complete -c spack -n '__fish_spack_using_command env loads' -s m -l module-type -r -f -a 'tcl lmod'
complete -c spack -n '__fish_spack_using_command env loads' -s m -l module-type -r -d 'type of module system to generate loads for'
complete -c spack -n '__fish_spack_using_command env loads' -l input-only -f -a shell
complete -c spack -n '__fish_spack_using_command env loads' -l input-only -d 'generate input for module command (instead of a shell script)'
complete -c spack -n '__fish_spack_using_command env loads' -s p -l prefix -r -f -a prefix
complete -c spack -n '__fish_spack_using_command env loads' -s p -l prefix -r -d 'prepend to module names when issuing module load commands'
complete -c spack -n '__fish_spack_using_command env loads' -s x -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command env loads' -s x -l exclude -r -d 'exclude package from output; may be specified multiple times'
complete -c spack -n '__fish_spack_using_command env loads' -s r -l dependencies -f -a recurse_dependencies
complete -c spack -n '__fish_spack_using_command env loads' -s r -l dependencies -d 'recursively traverse spec dependencies'

# spack env view
set -g __fish_spack_optspecs_spack_env_view h/help
complete -c spack -n '__fish_spack_using_command_pos 0 env view' -f -a 'regenerate enable disable'
complete -c spack -n '__fish_spack_using_command env view' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env view' -s h -l help -d 'show this help message and exit'

# spack env update
set -g __fish_spack_optspecs_spack_env_update h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos 0 env update' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env update' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env update' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env update' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env update' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack env revert
set -g __fish_spack_optspecs_spack_env_revert h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos 0 env revert' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env revert' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env revert' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env revert' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env revert' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack env depfile
set -g __fish_spack_optspecs_spack_env_depfile h/help make-prefix= make-disable-jobserver use-buildcache= o/output= G/generator=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 env depfile' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command env depfile' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env depfile' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env depfile' -l make-prefix -l make-target-prefix -r -f -a make_prefix
complete -c spack -n '__fish_spack_using_command env depfile' -l make-prefix -l make-target-prefix -r -d 'prefix Makefile targets/variables with <TARGET>/<name>,'
complete -c spack -n '__fish_spack_using_command env depfile' -l make-disable-jobserver -f -a jobserver
complete -c spack -n '__fish_spack_using_command env depfile' -l make-disable-jobserver -d 'disable POSIX jobserver support'
complete -c spack -n '__fish_spack_using_command env depfile' -l use-buildcache -r -f -a use_buildcache
complete -c spack -n '__fish_spack_using_command env depfile' -l use-buildcache -r -d 'use `only` to prune redundant build dependencies'
complete -c spack -n '__fish_spack_using_command env depfile' -s o -l output -r -f -a output
complete -c spack -n '__fish_spack_using_command env depfile' -s o -l output -r -d 'write the depfile to FILE rather than to stdout'
complete -c spack -n '__fish_spack_using_command env depfile' -s G -l generator -r -f -a make
complete -c spack -n '__fish_spack_using_command env depfile' -s G -l generator -r -d 'specify the depfile type (only supports `make`)'

# spack env track
set -g __fish_spack_optspecs_spack_env_track h/help n/name= y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos 0 env track' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env track' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env track' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env track' -s n -l name -r -f -a name
complete -c spack -n '__fish_spack_using_command env track' -s n -l name -r -d 'custom environment name'
complete -c spack -n '__fish_spack_using_command env track' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env track' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack env untrack
set -g __fish_spack_optspecs_spack_env_untrack h/help f/force y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 env untrack' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command env untrack' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command env untrack' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command env untrack' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command env untrack' -s f -l force -d 'force unlink even when environment is active'
complete -c spack -n '__fish_spack_using_command env untrack' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command env untrack' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack extensions
set -g __fish_spack_optspecs_spack_extensions h/help l/long L/very-long d/deps p/paths s/show=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 extensions' -f -a '(__fish_spack_extensions)'
complete -c spack -n '__fish_spack_using_command extensions' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command extensions' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command extensions' -s l -l long -f -a long
complete -c spack -n '__fish_spack_using_command extensions' -s l -l long -d 'show dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command extensions' -s L -l very-long -f -a very_long
complete -c spack -n '__fish_spack_using_command extensions' -s L -l very-long -d 'show full dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command extensions' -s d -l deps -f -a deps
complete -c spack -n '__fish_spack_using_command extensions' -s d -l deps -d 'output dependencies along with found specs'
complete -c spack -n '__fish_spack_using_command extensions' -s p -l paths -f -a paths
complete -c spack -n '__fish_spack_using_command extensions' -s p -l paths -d 'show paths to package install directories'
complete -c spack -n '__fish_spack_using_command extensions' -s s -l show -r -f -a 'packages installed all'
complete -c spack -n '__fish_spack_using_command extensions' -s s -l show -r -d 'show only part of output'

# spack external
set -g __fish_spack_optspecs_spack_external h/help
complete -c spack -n '__fish_spack_using_command_pos 0 external' -f -a find -d 'add external packages to packages.yaml'
complete -c spack -n '__fish_spack_using_command_pos 0 external' -f -a list -d 'list detectable packages, by repository and name'
complete -c spack -n '__fish_spack_using_command_pos 0 external' -f -a ls -d 'list detectable packages, by repository and name'
complete -c spack -n '__fish_spack_using_command_pos 0 external' -f -a read-cray-manifest -d 'consume a Spack-compatible description of externally-installed packages, including dependency relationships'
complete -c spack -n '__fish_spack_using_command external' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command external' -s h -l help -d 'show this help message and exit'

# spack external find
set -g __fish_spack_optspecs_spack_external_find h/help not-buildable exclude= p/path= scope= all t/tag= j/jobs=

complete -c spack -n '__fish_spack_using_command external find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command external find' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command external find' -l not-buildable -f -a not_buildable
complete -c spack -n '__fish_spack_using_command external find' -l not-buildable -d 'packages with detected externals won'"'"'t be built with Spack'
complete -c spack -n '__fish_spack_using_command external find' -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command external find' -l exclude -r -d 'packages to exclude from search'
complete -c spack -n '__fish_spack_using_command external find' -s p -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command external find' -s p -l path -r -d 'one or more alternative search paths for finding externals'
complete -c spack -n '__fish_spack_using_command external find' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command external find' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command external find' -l all -f -a all
complete -c spack -n '__fish_spack_using_command external find' -l all -d 'search for all packages that Spack knows about'
complete -c spack -n '__fish_spack_using_command external find' -s t -l tag -r -f -a tags
complete -c spack -n '__fish_spack_using_command external find' -s t -l tag -r -d 'filter a package query by tag (multiple use allowed)'
complete -c spack -n '__fish_spack_using_command external find' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command external find' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack external list
set -g __fish_spack_optspecs_spack_external_list h/help
complete -c spack -n '__fish_spack_using_command external list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command external list' -s h -l help -d 'show this help message and exit'

# spack external ls
set -g __fish_spack_optspecs_spack_external_ls h/help
complete -c spack -n '__fish_spack_using_command external ls' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command external ls' -s h -l help -d 'show this help message and exit'

# spack external read-cray-manifest
set -g __fish_spack_optspecs_spack_external_read_cray_manifest h/help file= directory= ignore-default-dir dry-run fail-on-error
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l file -r -f -a file
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l file -r -d 'specify a location other than the default'
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l directory -r -f -a directory
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l directory -r -d 'specify a directory storing a group of manifest files'
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l ignore-default-dir -f -a ignore_default_dir
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l ignore-default-dir -d 'ignore the default directory of manifest files'
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l dry-run -f -a dry_run
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l dry-run -d 'don'"'"'t modify DB with files that are read'
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l fail-on-error -f -a fail_on_error
complete -c spack -n '__fish_spack_using_command external read-cray-manifest' -l fail-on-error -d 'if a manifest file cannot be parsed, fail and report the full stack trace'

# spack fetch
set -g __fish_spack_optspecs_spack_fetch h/help n/no-checksum m/missing D/dependencies f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 fetch' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command fetch' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command fetch' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command fetch' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command fetch' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'
complete -c spack -n '__fish_spack_using_command fetch' -s m -l missing -f -a missing
complete -c spack -n '__fish_spack_using_command fetch' -s m -l missing -d 'fetch only missing (not yet installed) dependencies'
complete -c spack -n '__fish_spack_using_command fetch' -s D -l dependencies -f -a dependencies
complete -c spack -n '__fish_spack_using_command fetch' -s D -l dependencies -d 'also fetch all dependencies'
complete -c spack -n '__fish_spack_using_command fetch' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command fetch' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command fetch' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command fetch' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command fetch' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command fetch' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command fetch' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command fetch' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command fetch' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command fetch' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack find
set -g __fish_spack_optspecs_spack_find h/help format= H/hashes json I/install-status specfile-format d/deps p/paths groups no-groups l/long L/very-long t/tag= N/namespaces r/only-roots c/show-concretized f/show-flags show-full-compiler x/explicit X/implicit u/unknown m/missing v/variants loaded M/only-missing only-deprecated deprecated install-tree= start-date= end-date=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 find' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command find' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command find' -l format -r -f -a format
complete -c spack -n '__fish_spack_using_command find' -l format -r -d 'output specs with the specified format string'
complete -c spack -n '__fish_spack_using_command find' -s H -l hashes -f -a format
complete -c spack -n '__fish_spack_using_command find' -s H -l hashes -d 'same as '"'"'--format {/hash}'"'"'; use with xargs or $()'
complete -c spack -n '__fish_spack_using_command find' -l json -f -a json
complete -c spack -n '__fish_spack_using_command find' -l json -d 'output specs as machine-readable json records'
complete -c spack -n '__fish_spack_using_command find' -s I -l install-status -f -a install_status
complete -c spack -n '__fish_spack_using_command find' -s I -l install-status -d 'show install status of packages'
complete -c spack -n '__fish_spack_using_command find' -l specfile-format -f -a specfile_format
complete -c spack -n '__fish_spack_using_command find' -l specfile-format -d 'show the specfile format for installed deps '
complete -c spack -n '__fish_spack_using_command find' -s d -l deps -f -a deps
complete -c spack -n '__fish_spack_using_command find' -s d -l deps -d 'output dependencies along with found specs'
complete -c spack -n '__fish_spack_using_command find' -s p -l paths -f -a paths
complete -c spack -n '__fish_spack_using_command find' -s p -l paths -d 'show paths to package install directories'
complete -c spack -n '__fish_spack_using_command find' -l groups -f -a groups
complete -c spack -n '__fish_spack_using_command find' -l groups -d 'display specs in arch/compiler groups (default on)'
complete -c spack -n '__fish_spack_using_command find' -l no-groups -f -a groups
complete -c spack -n '__fish_spack_using_command find' -l no-groups -d 'do not group specs by arch/compiler'
complete -c spack -n '__fish_spack_using_command find' -s l -l long -f -a long
complete -c spack -n '__fish_spack_using_command find' -s l -l long -d 'show dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command find' -s L -l very-long -f -a very_long
complete -c spack -n '__fish_spack_using_command find' -s L -l very-long -d 'show full dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command find' -s t -l tag -r -f -a tags
complete -c spack -n '__fish_spack_using_command find' -s t -l tag -r -d 'filter a package query by tag (multiple use allowed)'
complete -c spack -n '__fish_spack_using_command find' -s N -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command find' -s N -l namespaces -d 'show fully qualified package names'
complete -c spack -n '__fish_spack_using_command find' -s r -l only-roots -f -a only_roots
complete -c spack -n '__fish_spack_using_command find' -s r -l only-roots -d 'don'"'"'t show full list of installed specs in an environment'
complete -c spack -n '__fish_spack_using_command find' -s c -l show-concretized -f -a show_concretized
complete -c spack -n '__fish_spack_using_command find' -s c -l show-concretized -d 'show concretized specs in an environment'
complete -c spack -n '__fish_spack_using_command find' -s f -l show-flags -f -a show_flags
complete -c spack -n '__fish_spack_using_command find' -s f -l show-flags -d 'show spec compiler flags'
complete -c spack -n '__fish_spack_using_command find' -l show-full-compiler -f -a show_full_compiler
complete -c spack -n '__fish_spack_using_command find' -l show-full-compiler -d '(DEPRECATED) show full compiler specs. Currently it'"'"'s a no-op'
complete -c spack -n '__fish_spack_using_command find' -s x -l explicit -f -a explicit
complete -c spack -n '__fish_spack_using_command find' -s x -l explicit -d 'show only specs that were installed explicitly'
complete -c spack -n '__fish_spack_using_command find' -s X -l implicit -f -a implicit
complete -c spack -n '__fish_spack_using_command find' -s X -l implicit -d 'show only specs that were installed as dependencies'
complete -c spack -n '__fish_spack_using_command find' -s u -l unknown -f -a unknown
complete -c spack -n '__fish_spack_using_command find' -s u -l unknown -d 'show only specs Spack does not have a package for'
complete -c spack -n '__fish_spack_using_command find' -s m -l missing -f -a missing
complete -c spack -n '__fish_spack_using_command find' -s m -l missing -d 'show missing dependencies as well as installed specs'
complete -c spack -n '__fish_spack_using_command find' -s v -l variants -f -a variants
complete -c spack -n '__fish_spack_using_command find' -s v -l variants -d 'show variants in output (can be long)'
complete -c spack -n '__fish_spack_using_command find' -l loaded -f -a loaded
complete -c spack -n '__fish_spack_using_command find' -l loaded -d 'show only packages loaded in the user environment'
complete -c spack -n '__fish_spack_using_command find' -s M -l only-missing -f -a only_missing
complete -c spack -n '__fish_spack_using_command find' -s M -l only-missing -d 'show only missing dependencies'
complete -c spack -n '__fish_spack_using_command find' -l only-deprecated -f -a only_deprecated
complete -c spack -n '__fish_spack_using_command find' -l only-deprecated -d 'show only deprecated packages'
complete -c spack -n '__fish_spack_using_command find' -l deprecated -f -a deprecated
complete -c spack -n '__fish_spack_using_command find' -l deprecated -d 'show deprecated packages as well as installed specs'
complete -c spack -n '__fish_spack_using_command find' -l install-tree -r -f -a install_tree
complete -c spack -n '__fish_spack_using_command find' -l install-tree -r -d 'Install trees to query: '"'"'all'"'"' (default), '"'"'local'"'"', '"'"'upstream'"'"', upstream name or path'
complete -c spack -n '__fish_spack_using_command find' -l start-date -r -f -a start_date
complete -c spack -n '__fish_spack_using_command find' -l start-date -r -d 'earliest date of installation [YYYY-MM-DD]'
complete -c spack -n '__fish_spack_using_command find' -l end-date -r -f -a end_date
complete -c spack -n '__fish_spack_using_command find' -l end-date -r -d 'latest date of installation [YYYY-MM-DD]'

# spack gc
set -g __fish_spack_optspecs_spack_gc h/help E/except-any-environment e/except-environment= b/keep-build-dependencies y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 gc' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command gc' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gc' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gc' -s E -l except-any-environment -f -a except_any_environment
complete -c spack -n '__fish_spack_using_command gc' -s E -l except-any-environment -d 'remove everything unless needed by an environment'
complete -c spack -n '__fish_spack_using_command gc' -s e -l except-environment -r -f -a except_environment
complete -c spack -n '__fish_spack_using_command gc' -s e -l except-environment -r -d 'remove everything unless needed by specified environment'
complete -c spack -n '__fish_spack_using_command gc' -s b -l keep-build-dependencies -f -a keep_build_dependencies
complete -c spack -n '__fish_spack_using_command gc' -s b -l keep-build-dependencies -d 'do not remove installed build-only dependencies of roots'
complete -c spack -n '__fish_spack_using_command gc' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command gc' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack gpg
set -g __fish_spack_optspecs_spack_gpg h/help
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a verify -d 'verify a signed package'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a trust -d 'add a key to the keyring'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a untrust -d 'remove a key from the keyring'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a sign -d 'sign a package'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a create -d 'create a new key'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a list -d 'list keys available in the keyring'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a init -d 'add the default keys to the keyring'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a export -d 'export a gpg key, optionally including secret key'
complete -c spack -n '__fish_spack_using_command_pos 0 gpg' -f -a publish -d 'publish public keys to a build cache'
complete -c spack -n '__fish_spack_using_command gpg' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg' -s h -l help -d 'show this help message and exit'

# spack gpg verify
set -g __fish_spack_optspecs_spack_gpg_verify h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 gpg verify' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command gpg verify' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg verify' -s h -l help -d 'show this help message and exit'

# spack gpg trust
set -g __fish_spack_optspecs_spack_gpg_trust h/help

complete -c spack -n '__fish_spack_using_command gpg trust' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg trust' -s h -l help -d 'show this help message and exit'

# spack gpg untrust
set -g __fish_spack_optspecs_spack_gpg_untrust h/help signing
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 gpg untrust' -f -a '(__fish_spack_gpg_keys)'
complete -c spack -n '__fish_spack_using_command gpg untrust' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg untrust' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg untrust' -l signing -f -a signing
complete -c spack -n '__fish_spack_using_command gpg untrust' -l signing -d 'allow untrusting signing keys'

# spack gpg sign
set -g __fish_spack_optspecs_spack_gpg_sign h/help output= key= clearsign
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 gpg sign' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command gpg sign' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg sign' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg sign' -l output -r -f -a output
complete -c spack -n '__fish_spack_using_command gpg sign' -l output -r -d 'the directory to place signatures'
complete -c spack -n '__fish_spack_using_command gpg sign' -l key -r -f -a key
complete -c spack -n '__fish_spack_using_command gpg sign' -l key -r -d 'the key to use for signing'
complete -c spack -n '__fish_spack_using_command gpg sign' -l clearsign -f -a clearsign
complete -c spack -n '__fish_spack_using_command gpg sign' -l clearsign -d 'if specified, create a clearsign signature'

# spack gpg create
set -g __fish_spack_optspecs_spack_gpg_create h/help comment= expires= export= export-secret=

complete -c spack -n '__fish_spack_using_command gpg create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg create' -l comment -r -f -a comment
complete -c spack -n '__fish_spack_using_command gpg create' -l comment -r -d 'a description for the intended use of the key'
complete -c spack -n '__fish_spack_using_command gpg create' -l expires -r -f -a expires
complete -c spack -n '__fish_spack_using_command gpg create' -l expires -r -d 'when the key should expire'
complete -c spack -n '__fish_spack_using_command gpg create' -l export -r -f -a export
complete -c spack -n '__fish_spack_using_command gpg create' -l export -r -d 'export the public key to a file'
complete -c spack -n '__fish_spack_using_command gpg create' -l export-secret -r -f -a secret
complete -c spack -n '__fish_spack_using_command gpg create' -l export-secret -r -d 'export the private key to a file'

# spack gpg list
set -g __fish_spack_optspecs_spack_gpg_list h/help trusted signing
complete -c spack -n '__fish_spack_using_command gpg list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg list' -l trusted -f -a trusted
complete -c spack -n '__fish_spack_using_command gpg list' -l trusted -d 'list trusted keys'
complete -c spack -n '__fish_spack_using_command gpg list' -l signing -f -a signing
complete -c spack -n '__fish_spack_using_command gpg list' -l signing -d 'list keys which may be used for signing'

# spack gpg init
set -g __fish_spack_optspecs_spack_gpg_init h/help from=
complete -c spack -n '__fish_spack_using_command gpg init' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg init' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg init' -l from -r -f -a import_dir

# spack gpg export
set -g __fish_spack_optspecs_spack_gpg_export h/help secret
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 gpg export' -f -a '(__fish_spack_gpg_keys)'
complete -c spack -n '__fish_spack_using_command gpg export' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg export' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg export' -l secret -f -a secret
complete -c spack -n '__fish_spack_using_command gpg export' -l secret -d 'export secret keys'

# spack gpg publish
set -g __fish_spack_optspecs_spack_gpg_publish h/help d/directory= m/mirror-name= mirror-url= update-index
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 gpg publish' -f -a '(__fish_spack_gpg_keys)'
complete -c spack -n '__fish_spack_using_command gpg publish' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command gpg publish' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command gpg publish' -s d -l directory -r -f -a directory
complete -c spack -n '__fish_spack_using_command gpg publish' -s d -l directory -r -d 'local directory where keys will be published'
complete -c spack -n '__fish_spack_using_command gpg publish' -s m -l mirror-name -r -f -a mirror_name
complete -c spack -n '__fish_spack_using_command gpg publish' -s m -l mirror-name -r -d 'name of the mirror where keys will be published'
complete -c spack -n '__fish_spack_using_command gpg publish' -l mirror-url -r -f -a mirror_url
complete -c spack -n '__fish_spack_using_command gpg publish' -l mirror-url -r -d 'URL of the mirror where keys will be published'
complete -c spack -n '__fish_spack_using_command gpg publish' -l update-index -l rebuild-index -f -a update_index
complete -c spack -n '__fish_spack_using_command gpg publish' -l update-index -l rebuild-index -d 'regenerate buildcache key index after publishing key(s)'

# spack graph
set -g __fish_spack_optspecs_spack_graph h/help a/ascii d/dot s/static c/color i/installed deptype=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 graph' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command graph' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command graph' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command graph' -s a -l ascii -f -a ascii
complete -c spack -n '__fish_spack_using_command graph' -s a -l ascii -d 'draw graph as ascii to stdout (default)'
complete -c spack -n '__fish_spack_using_command graph' -s d -l dot -f -a dot
complete -c spack -n '__fish_spack_using_command graph' -s d -l dot -d 'generate graph in dot format and print to stdout'
complete -c spack -n '__fish_spack_using_command graph' -s s -l static -f -a static
complete -c spack -n '__fish_spack_using_command graph' -s s -l static -d 'graph static (possible) deps, don'"'"'t concretize (implies --dot)'
complete -c spack -n '__fish_spack_using_command graph' -s c -l color -f -a color
complete -c spack -n '__fish_spack_using_command graph' -s c -l color -d 'use different colors for different dependency types'
complete -c spack -n '__fish_spack_using_command graph' -s i -l installed -f -a installed
complete -c spack -n '__fish_spack_using_command graph' -s i -l installed -d 'graph specs from the DB'
complete -c spack -n '__fish_spack_using_command graph' -l deptype -r -f -a deptype
complete -c spack -n '__fish_spack_using_command graph' -l deptype -r -d 'comma-separated list of deptypes to traverse (default=build,link,run,test)'

# spack help
set -g __fish_spack_optspecs_spack_help h/help a/all spec
complete -c spack -n '__fish_spack_using_command_pos 0 help' -f -a '(__fish_spack_commands)'
complete -c spack -n '__fish_spack_using_command help' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command help' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command help' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command help' -s a -l all -d 'list all available commands and options'
complete -c spack -n '__fish_spack_using_command help' -l spec -f -a guide
complete -c spack -n '__fish_spack_using_command help' -l spec -d 'help on the package specification syntax'

# spack info
set -g __fish_spack_optspecs_spack_info h/help a/all detectable maintainers namespace no-dependencies no-variants no-versions phases tags tests virtuals variants-by-name
complete -c spack -n '__fish_spack_using_command_pos 0 info' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command info' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command info' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command info' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command info' -s a -l all -d 'output all package information'
complete -c spack -n '__fish_spack_using_command info' -l detectable -f -a detectable
complete -c spack -n '__fish_spack_using_command info' -l detectable -d 'output information on external detection'
complete -c spack -n '__fish_spack_using_command info' -l maintainers -f -a maintainers
complete -c spack -n '__fish_spack_using_command info' -l maintainers -d 'output package maintainers'
complete -c spack -n '__fish_spack_using_command info' -l namespace -f -a namespace
complete -c spack -n '__fish_spack_using_command info' -l namespace -d 'output package namespace'
complete -c spack -n '__fish_spack_using_command info' -l no-dependencies -f -a no_dependencies
complete -c spack -n '__fish_spack_using_command info' -l no-dependencies -d 'do not output build, link, and run package dependencies'
complete -c spack -n '__fish_spack_using_command info' -l no-variants -f -a no_variants
complete -c spack -n '__fish_spack_using_command info' -l no-variants -d 'do not output variants'
complete -c spack -n '__fish_spack_using_command info' -l no-versions -f -a no_versions
complete -c spack -n '__fish_spack_using_command info' -l no-versions -d 'do not output versions'
complete -c spack -n '__fish_spack_using_command info' -l phases -f -a phases
complete -c spack -n '__fish_spack_using_command info' -l phases -d 'output installation phases'
complete -c spack -n '__fish_spack_using_command info' -l tags -f -a tags
complete -c spack -n '__fish_spack_using_command info' -l tags -d 'output package tags'
complete -c spack -n '__fish_spack_using_command info' -l tests -f -a tests
complete -c spack -n '__fish_spack_using_command info' -l tests -d 'output relevant build-time and stand-alone tests'
complete -c spack -n '__fish_spack_using_command info' -l virtuals -f -a virtuals
complete -c spack -n '__fish_spack_using_command info' -l virtuals -d 'output virtual packages'
complete -c spack -n '__fish_spack_using_command info' -l variants-by-name -f -a variants_by_name
complete -c spack -n '__fish_spack_using_command info' -l variants-by-name -d 'list variants in strict name order; don'"'"'t group by condition'

# spack install
set -g __fish_spack_optspecs_spack_install h/help only= u/until= p/concurrent-packages= j/jobs= overwrite fail-fast keep-prefix keep-stage dont-restage use-cache no-cache cache-only use-buildcache= include-build-deps no-check-signature show-log-on-error source n/no-checksum v/verbose fake only-concrete add no-add clean dirty test= log-format= log-file= help-cdash cdash-upload-url= cdash-build= cdash-site= cdash-track= cdash-buildstamp= y/yes-to-all f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 install' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command install' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command install' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command install' -l only -r -f -a 'package dependencies'
complete -c spack -n '__fish_spack_using_command install' -l only -r -d 'select the mode of installation'
complete -c spack -n '__fish_spack_using_command install' -s u -l until -r -f -a until
complete -c spack -n '__fish_spack_using_command install' -s u -l until -r -d 'phase to stop after when installing (default None)'
complete -c spack -n '__fish_spack_using_command install' -s p -l concurrent-packages -r -f -a concurrent_packages
complete -c spack -n '__fish_spack_using_command install' -s p -l concurrent-packages -r -d 'maximum number of packages to build concurrently'
complete -c spack -n '__fish_spack_using_command install' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command install' -s j -l jobs -r -d 'explicitly set number of parallel jobs'
complete -c spack -n '__fish_spack_using_command install' -l overwrite -f -a overwrite
complete -c spack -n '__fish_spack_using_command install' -l overwrite -d 'reinstall an existing spec, even if it has dependents'
complete -c spack -n '__fish_spack_using_command install' -l fail-fast -f -a fail_fast
complete -c spack -n '__fish_spack_using_command install' -l fail-fast -d 'stop all builds if any build fails (default is best effort)'
complete -c spack -n '__fish_spack_using_command install' -l keep-prefix -f -a keep_prefix
complete -c spack -n '__fish_spack_using_command install' -l keep-prefix -d 'don'"'"'t remove the install prefix if installation fails'
complete -c spack -n '__fish_spack_using_command install' -l keep-stage -f -a keep_stage
complete -c spack -n '__fish_spack_using_command install' -l keep-stage -d 'don'"'"'t remove the build stage if installation succeeds'
complete -c spack -n '__fish_spack_using_command install' -l dont-restage -f -a dont_restage
complete -c spack -n '__fish_spack_using_command install' -l dont-restage -d 'if a partial install is detected, don'"'"'t delete prior state'
complete -c spack -n '__fish_spack_using_command install' -l use-cache -f -a use_cache
complete -c spack -n '__fish_spack_using_command install' -l use-cache -d 'check for pre-built Spack packages in mirrors (default)'
complete -c spack -n '__fish_spack_using_command install' -l no-cache -f -a use_cache
complete -c spack -n '__fish_spack_using_command install' -l no-cache -d 'do not check for pre-built Spack packages in mirrors'
complete -c spack -n '__fish_spack_using_command install' -l cache-only -f -a cache_only
complete -c spack -n '__fish_spack_using_command install' -l cache-only -d 'only install package from binary mirrors'
complete -c spack -n '__fish_spack_using_command install' -l use-buildcache -r -f -a use_buildcache
complete -c spack -n '__fish_spack_using_command install' -l use-buildcache -r -d 'select the mode of buildcache for the '"'"'package'"'"' and '"'"'dependencies'"'"''
complete -c spack -n '__fish_spack_using_command install' -l include-build-deps -f -a include_build_deps
complete -c spack -n '__fish_spack_using_command install' -l include-build-deps -d 'include build deps when installing from cache, useful for CI pipeline troubleshooting'
complete -c spack -n '__fish_spack_using_command install' -l no-check-signature -f -a unsigned
complete -c spack -n '__fish_spack_using_command install' -l no-check-signature -d 'do not check signatures of binary packages (override mirror config)'
complete -c spack -n '__fish_spack_using_command install' -l show-log-on-error -f -a show_log_on_error
complete -c spack -n '__fish_spack_using_command install' -l show-log-on-error -d 'print full build log to stderr if build fails'
complete -c spack -n '__fish_spack_using_command install' -l source -f -a install_source
complete -c spack -n '__fish_spack_using_command install' -l source -d 'install source files in prefix'
complete -c spack -n '__fish_spack_using_command install' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command install' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'
complete -c spack -n '__fish_spack_using_command install' -s v -l verbose -f -a install_verbose
complete -c spack -n '__fish_spack_using_command install' -s v -l verbose -d 'display verbose build output while installing'
complete -c spack -n '__fish_spack_using_command install' -l fake -f -a fake
complete -c spack -n '__fish_spack_using_command install' -l fake -d 'fake install for debug purposes'
complete -c spack -n '__fish_spack_using_command install' -l only-concrete -f -a only_concrete
complete -c spack -n '__fish_spack_using_command install' -l only-concrete -d '(with environment) only install already concretized specs'
complete -c spack -n '__fish_spack_using_command install' -l add -f -a add
complete -c spack -n '__fish_spack_using_command install' -l add -d '(with environment) add spec to the environment as a root'
complete -c spack -n '__fish_spack_using_command install' -l no-add -f -a add
complete -c spack -n '__fish_spack_using_command install' -l no-add -d '(with environment) do not add spec to the environment as a root'
complete -c spack -n '__fish_spack_using_command install' -l clean -f -a dirty
complete -c spack -n '__fish_spack_using_command install' -l clean -d 'unset harmful variables in the build environment (default)'
complete -c spack -n '__fish_spack_using_command install' -l dirty -f -a dirty
complete -c spack -n '__fish_spack_using_command install' -l dirty -d 'preserve user environment in spack'"'"'s build environment (danger!)'
complete -c spack -n '__fish_spack_using_command install' -l test -r -f -a 'root all'
complete -c spack -n '__fish_spack_using_command install' -l test -r -d 'run tests on only root packages or all packages'
complete -c spack -n '__fish_spack_using_command install' -l log-format -r -f -a 'junit cdash'
complete -c spack -n '__fish_spack_using_command install' -l log-format -r -d 'format to be used for log files'
complete -c spack -n '__fish_spack_using_command install' -l log-file -r -f -a log_file
complete -c spack -n '__fish_spack_using_command install' -l log-file -r -d 'filename for the log file'
complete -c spack -n '__fish_spack_using_command install' -l help-cdash -f -a help_cdash
complete -c spack -n '__fish_spack_using_command install' -l help-cdash -d 'show usage instructions for CDash reporting'
complete -c spack -n '__fish_spack_using_command install' -l cdash-upload-url -r -f -a cdash_upload_url
complete -c spack -n '__fish_spack_using_command install' -l cdash-build -r -f -a cdash_build
complete -c spack -n '__fish_spack_using_command install' -l cdash-site -r -f -a cdash_site
complete -c spack -n '__fish_spack_using_command install' -l cdash-track -r -f -a cdash_track
complete -c spack -n '__fish_spack_using_command install' -l cdash-buildstamp -r -f -a cdash_buildstamp
complete -c spack -n '__fish_spack_using_command install' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command install' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command install' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command install' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command install' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command install' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command install' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command install' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command install' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command install' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command install' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command install' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack license
set -g __fish_spack_optspecs_spack_license h/help root=
complete -c spack -n '__fish_spack_using_command_pos 0 license' -f -a list-files -d 'list files in spack that should have license headers'
complete -c spack -n '__fish_spack_using_command_pos 0 license' -f -a verify -d 'verify that files in spack have the right license header'
complete -c spack -n '__fish_spack_using_command license' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command license' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command license' -l root -r -f -a root
complete -c spack -n '__fish_spack_using_command license' -l root -r -d 'scan a different prefix for license issues'

# spack license list-files
set -g __fish_spack_optspecs_spack_license_list_files h/help
complete -c spack -n '__fish_spack_using_command license list-files' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command license list-files' -s h -l help -d 'show this help message and exit'

# spack license verify
set -g __fish_spack_optspecs_spack_license_verify h/help
complete -c spack -n '__fish_spack_using_command license verify' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command license verify' -s h -l help -d 'show this help message and exit'

# spack list
set -g __fish_spack_optspecs_spack_list h/help r/repo= d/search-description format= v/virtuals t/tag= count update=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 list' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command list' -s r -l repo -s N -l namespace -r -f -a repos
complete -c spack -n '__fish_spack_using_command list' -s r -l repo -s N -l namespace -r -d 'only list packages from the specified repo/namespace'
complete -c spack -n '__fish_spack_using_command list' -s d -l search-description -f -a search_description
complete -c spack -n '__fish_spack_using_command list' -s d -l search-description -d 'filtering will also search the description for a match'
complete -c spack -n '__fish_spack_using_command list' -l format -r -f -a 'name_only version_json html'
complete -c spack -n '__fish_spack_using_command list' -l format -r -d 'format to be used to print the output [default: name_only]'
complete -c spack -n '__fish_spack_using_command list' -s v -l virtuals -f -a virtuals
complete -c spack -n '__fish_spack_using_command list' -s v -l virtuals -d 'include virtual packages in list'
complete -c spack -n '__fish_spack_using_command list' -s t -l tag -r -f -a tags
complete -c spack -n '__fish_spack_using_command list' -s t -l tag -r -d 'filter a package query by tag (multiple use allowed)'
complete -c spack -n '__fish_spack_using_command list' -l count -f -a count
complete -c spack -n '__fish_spack_using_command list' -l count -d 'display the number of packages that would be listed'
complete -c spack -n '__fish_spack_using_command list' -l update -r -f -a update
complete -c spack -n '__fish_spack_using_command list' -l update -r -d 'write output to the specified file, if any package is newer'

# spack load
set -g __fish_spack_optspecs_spack_load h/help sh csh fish bat pwsh first list
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 load' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command load' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command load' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command load' -l sh -f -a shell
complete -c spack -n '__fish_spack_using_command load' -l sh -d 'print sh commands to load the package'
complete -c spack -n '__fish_spack_using_command load' -l csh -f -a shell
complete -c spack -n '__fish_spack_using_command load' -l csh -d 'print csh commands to load the package'
complete -c spack -n '__fish_spack_using_command load' -l fish -f -a shell
complete -c spack -n '__fish_spack_using_command load' -l fish -d 'print fish commands to load the package'
complete -c spack -n '__fish_spack_using_command load' -l bat -f -a shell
complete -c spack -n '__fish_spack_using_command load' -l bat -d 'print bat commands to load the package'
complete -c spack -n '__fish_spack_using_command load' -l pwsh -f -a shell
complete -c spack -n '__fish_spack_using_command load' -l pwsh -d 'print pwsh commands to load the package'
complete -c spack -n '__fish_spack_using_command load' -l first -f -a load_first
complete -c spack -n '__fish_spack_using_command load' -l first -d 'load the first match if multiple packages match the spec'
complete -c spack -n '__fish_spack_using_command load' -l list -f -a list
complete -c spack -n '__fish_spack_using_command load' -l list -d 'show loaded packages: same as `spack find --loaded`'

# spack location
set -g __fish_spack_optspecs_spack_location h/help m/module-dir r/spack-root i/install-dir p/package-dir repo= s/stage-dir S/stages c/source-dir b/build-dir e/env= first
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 location' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command location' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command location' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command location' -s m -l module-dir -f -a module_dir
complete -c spack -n '__fish_spack_using_command location' -s m -l module-dir -d 'spack python module directory'
complete -c spack -n '__fish_spack_using_command location' -s r -l spack-root -f -a spack_root
complete -c spack -n '__fish_spack_using_command location' -s r -l spack-root -d 'spack installation root'
complete -c spack -n '__fish_spack_using_command location' -s i -l install-dir -f -a install_dir
complete -c spack -n '__fish_spack_using_command location' -s i -l install-dir -d 'install prefix for spec (spec need not be installed)'
complete -c spack -n '__fish_spack_using_command location' -s p -l package-dir -f -a package_dir
complete -c spack -n '__fish_spack_using_command location' -s p -l package-dir -d 'directory enclosing a spec'"'"'s package.py file'
complete -c spack -n '__fish_spack_using_command location' -l repo -l packages -s P -r -f -a repo
complete -c spack -n '__fish_spack_using_command location' -l repo -l packages -s P -r -d 'package repository root (defaults to first configured repository)'
complete -c spack -n '__fish_spack_using_command location' -s s -l stage-dir -f -a stage_dir
complete -c spack -n '__fish_spack_using_command location' -s s -l stage-dir -d 'stage directory for a spec'
complete -c spack -n '__fish_spack_using_command location' -s S -l stages -f -a stages
complete -c spack -n '__fish_spack_using_command location' -s S -l stages -d 'top level stage directory'
complete -c spack -n '__fish_spack_using_command location' -s c -l source-dir -f -a source_dir
complete -c spack -n '__fish_spack_using_command location' -s c -l source-dir -d 'source directory for a spec (requires it to be staged first)'
complete -c spack -n '__fish_spack_using_command location' -s b -l build-dir -f -a build_dir
complete -c spack -n '__fish_spack_using_command location' -s b -l build-dir -d 'build directory for a spec (requires it to be staged first)'
complete -c spack -n '__fish_spack_using_command location' -s e -l env -r -f -a location_env
complete -c spack -n '__fish_spack_using_command location' -s e -l env -r -d 'location of the named or current environment'
complete -c spack -n '__fish_spack_using_command location' -l first -f -a find_first
complete -c spack -n '__fish_spack_using_command location' -l first -d 'use the first match if multiple packages match the spec'

# spack log-parse
set -g __fish_spack_optspecs_spack_log_parse h/help show= c/context= p/profile w/width= j/jobs=

complete -c spack -n '__fish_spack_using_command log-parse' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command log-parse' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command log-parse' -l show -r -f -a show
complete -c spack -n '__fish_spack_using_command log-parse' -l show -r -d 'comma-separated list of what to show; options: errors, warnings'
complete -c spack -n '__fish_spack_using_command log-parse' -s c -l context -r -f -a context
complete -c spack -n '__fish_spack_using_command log-parse' -s c -l context -r -d 'lines of context to show around lines of interest'
complete -c spack -n '__fish_spack_using_command log-parse' -s p -l profile -f -a profile
complete -c spack -n '__fish_spack_using_command log-parse' -s p -l profile -d 'print out a profile of time spent in regexes during parse'
complete -c spack -n '__fish_spack_using_command log-parse' -s w -l width -r -f -a width
complete -c spack -n '__fish_spack_using_command log-parse' -s w -l width -r -d 'wrap width: auto-size to terminal by default; 0 for no wrap'
complete -c spack -n '__fish_spack_using_command log-parse' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command log-parse' -s j -l jobs -r -d 'number of jobs to parse log file (default: 1 for short logs, ncpus for long logs)'

# spack logs
set -g __fish_spack_optspecs_spack_logs h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 logs' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command logs' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command logs' -s h -l help -d 'show this help message and exit'

# spack maintainers
set -g __fish_spack_optspecs_spack_maintainers h/help maintained unmaintained a/all by-user
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 maintainers' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command maintainers' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command maintainers' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command maintainers' -l maintained -f -a maintained
complete -c spack -n '__fish_spack_using_command maintainers' -l maintained -d 'show names of maintained packages'
complete -c spack -n '__fish_spack_using_command maintainers' -l unmaintained -f -a unmaintained
complete -c spack -n '__fish_spack_using_command maintainers' -l unmaintained -d 'show names of unmaintained packages'
complete -c spack -n '__fish_spack_using_command maintainers' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command maintainers' -s a -l all -d 'show maintainers for all packages'
complete -c spack -n '__fish_spack_using_command maintainers' -l by-user -f -a by_user
complete -c spack -n '__fish_spack_using_command maintainers' -l by-user -d 'show packages for users instead of users for packages'

# spack make-installer
set -g __fish_spack_optspecs_spack_make_installer h/help v/spack-version= s/spack-source= g/git-installer-verbosity=
complete -c spack -n '__fish_spack_using_command_pos 0 make-installer' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command make-installer' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command make-installer' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command make-installer' -s v -l spack-version -r -f -a spack_version
complete -c spack -n '__fish_spack_using_command make-installer' -s v -l spack-version -r -d 'download given spack version'
complete -c spack -n '__fish_spack_using_command make-installer' -s s -l spack-source -r -f -a spack_source
complete -c spack -n '__fish_spack_using_command make-installer' -s s -l spack-source -r -d 'full path to spack source'
complete -c spack -n '__fish_spack_using_command make-installer' -s g -l git-installer-verbosity -r -f -a 'SILENT VERYSILENT'
complete -c spack -n '__fish_spack_using_command make-installer' -s g -l git-installer-verbosity -r -d 'level of verbosity provided by bundled git installer (default is fully verbose)'

# spack mark
set -g __fish_spack_optspecs_spack_mark h/help a/all e/explicit i/implicit
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 mark' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command mark' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mark' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mark' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command mark' -s a -l all -d 'mark ALL installed packages that match each supplied spec'
complete -c spack -n '__fish_spack_using_command mark' -s e -l explicit -f -a explicit
complete -c spack -n '__fish_spack_using_command mark' -s e -l explicit -d 'mark packages as explicitly installed'
complete -c spack -n '__fish_spack_using_command mark' -s i -l implicit -f -a implicit
complete -c spack -n '__fish_spack_using_command mark' -s i -l implicit -d 'mark packages as implicitly installed'

# spack mirror
set -g __fish_spack_optspecs_spack_mirror h/help n/no-checksum
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a create -d 'create a directory to be used as a spack mirror, and fill it with package archives'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a destroy -d 'given a url, recursively delete everything under it'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a add -d 'add a mirror to Spack'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a remove -d 'remove a mirror by name'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a rm -d 'remove a mirror by name'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a set-url -d 'change the URL of a mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a set -d 'configure the connection details of a mirror'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a list -d 'print out available mirrors to the console'
complete -c spack -n '__fish_spack_using_command_pos 0 mirror' -f -a ls -d 'print out available mirrors to the console'
complete -c spack -n '__fish_spack_using_command mirror' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command mirror' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'

# spack mirror create
set -g __fish_spack_optspecs_spack_mirror_create h/help d/directory= a/all file= exclude-file= exclude-specs= skip-unstable-versions D/dependencies n/versions-per-spec= private f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 mirror create' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command mirror create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror create' -s d -l directory -r -f -a directory
complete -c spack -n '__fish_spack_using_command mirror create' -s d -l directory -r -d 'directory in which to create mirror'
complete -c spack -n '__fish_spack_using_command mirror create' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command mirror create' -s a -l all -d 'mirror all versions of all packages in Spack, or all packages in the current environment if there is an active environment (this requires significant time and space)'
complete -c spack -n '__fish_spack_using_command mirror create' -l file -r -f -a file
complete -c spack -n '__fish_spack_using_command mirror create' -l file -r -d 'file with specs of packages to put in mirror'
complete -c spack -n '__fish_spack_using_command mirror create' -l exclude-file -r -f -a exclude_file
complete -c spack -n '__fish_spack_using_command mirror create' -l exclude-file -r -d 'specs which Spack should not try to add to a mirror (listed in a file, one per line)'
complete -c spack -n '__fish_spack_using_command mirror create' -l exclude-specs -r -f -a exclude_specs
complete -c spack -n '__fish_spack_using_command mirror create' -l exclude-specs -r -d 'specs which Spack should not try to add to a mirror (specified on command line)'
complete -c spack -n '__fish_spack_using_command mirror create' -l skip-unstable-versions -f -a skip_unstable_versions
complete -c spack -n '__fish_spack_using_command mirror create' -l skip-unstable-versions -d 'don'"'"'t cache versions unless they identify a stable (unchanging) source code'
complete -c spack -n '__fish_spack_using_command mirror create' -s D -l dependencies -f -a dependencies
complete -c spack -n '__fish_spack_using_command mirror create' -s D -l dependencies -d 'also fetch all dependencies'
complete -c spack -n '__fish_spack_using_command mirror create' -s n -l versions-per-spec -r -f -a versions_per_spec
complete -c spack -n '__fish_spack_using_command mirror create' -s n -l versions-per-spec -r -d 'the number of versions to fetch for each spec, choose '"'"'all'"'"' to retrieve all versions of each package'
complete -c spack -n '__fish_spack_using_command mirror create' -l private -f -a private
complete -c spack -n '__fish_spack_using_command mirror create' -l private -d 'for a private mirror, include non-redistributable packages'
complete -c spack -n '__fish_spack_using_command mirror create' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command mirror create' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command mirror create' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command mirror create' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command mirror create' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command mirror create' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command mirror create' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command mirror create' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command mirror create' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command mirror create' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack mirror destroy
set -g __fish_spack_optspecs_spack_mirror_destroy h/help m/mirror-name= mirror-url=
complete -c spack -n '__fish_spack_using_command mirror destroy' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror destroy' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror destroy' -s m -l mirror-name -r -f -a mirror_name
complete -c spack -n '__fish_spack_using_command mirror destroy' -s m -l mirror-name -r -d 'find mirror to destroy by name'
complete -c spack -n '__fish_spack_using_command mirror destroy' -l mirror-url -r -f -a mirror_url
complete -c spack -n '__fish_spack_using_command mirror destroy' -l mirror-url -r -d 'find mirror to destroy by url'

# spack mirror add
set -g __fish_spack_optspecs_spack_mirror_add h/help scope= type= autopush unsigned signed s3-access-key-id= s3-access-key-id-variable= s3-access-key-secret= s3-access-key-secret-variable= s3-access-token= s3-access-token-variable= s3-profile= s3-endpoint-url= oci-username= oci-username-variable= oci-password= oci-password-variable=
complete -c spack -n '__fish_spack_using_command_pos 0 mirror add' -f
complete -c spack -n '__fish_spack_using_command mirror add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror add' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror add' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command mirror add' -l type -r -f -a 'binary source'
complete -c spack -n '__fish_spack_using_command mirror add' -l type -r -d 'specify the mirror type: for both binary and source use `--type binary --type source` (default)'
complete -c spack -n '__fish_spack_using_command mirror add' -l autopush -f -a autopush
complete -c spack -n '__fish_spack_using_command mirror add' -l autopush -d 'set mirror to push automatically after installation'
complete -c spack -n '__fish_spack_using_command mirror add' -l unsigned -f -a signed
complete -c spack -n '__fish_spack_using_command mirror add' -l unsigned -d 'do not require signing and signature verification when pushing and installing from this build cache'
complete -c spack -n '__fish_spack_using_command mirror add' -l signed -f -a signed
complete -c spack -n '__fish_spack_using_command mirror add' -l signed -d 'require signing and signature verification when pushing and installing from this build cache'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-id -r -f -a s3_access_key_id
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-id -r -d 'ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-id-variable -r -f -a s3_access_key_id_variable
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-id-variable -r -d 'environment variable containing ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-secret -r -f -a s3_access_key_secret
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-secret -r -d 'secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-secret-variable -r -f -a s3_access_key_secret_variable
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-key-secret-variable -r -d 'environment variable containing secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-token -r -f -a s3_access_token
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-token -r -d 'access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-token-variable -r -f -a s3_access_token_variable
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-access-token-variable -r -d 'environment variable containing access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-profile -r -f -a s3_profile
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-profile -r -d 'S3 profile name to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-endpoint-url -r -f -a s3_endpoint_url
complete -c spack -n '__fish_spack_using_command mirror add' -l s3-endpoint-url -r -d 'endpoint URL to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-username -r -f -a oci_username
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-username -r -d 'username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-username-variable -r -f -a oci_username_variable
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-username-variable -r -d 'environment variable containing username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-password -r -f -a oci_password
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-password -r -d 'password to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-password-variable -r -f -a oci_password_variable
complete -c spack -n '__fish_spack_using_command mirror add' -l oci-password-variable -r -d 'environment variable containing password to use to connect to this OCI mirror'

# spack mirror remove
set -g __fish_spack_optspecs_spack_mirror_remove h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 mirror remove' -f -a '(__fish_spack_mirrors)'
complete -c spack -n '__fish_spack_using_command mirror remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror remove' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror remove' -l scope -r -d 'configuration scope to modify'

# spack mirror rm
set -g __fish_spack_optspecs_spack_mirror_rm h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 mirror rm' -f -a '(__fish_spack_mirrors)'
complete -c spack -n '__fish_spack_using_command mirror rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror rm' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror rm' -l scope -r -d 'configuration scope to modify'

# spack mirror set-url
set -g __fish_spack_optspecs_spack_mirror_set_url h/help push fetch scope= s3-access-key-id= s3-access-key-id-variable= s3-access-key-secret= s3-access-key-secret-variable= s3-access-token= s3-access-token-variable= s3-profile= s3-endpoint-url= oci-username= oci-username-variable= oci-password= oci-password-variable=
complete -c spack -n '__fish_spack_using_command_pos 0 mirror set-url' -f -a '(__fish_spack_mirrors)'
complete -c spack -n '__fish_spack_using_command mirror set-url' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror set-url' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l push -f -a push
complete -c spack -n '__fish_spack_using_command mirror set-url' -l push -d 'set only the URL used for uploading'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l fetch -f -a fetch
complete -c spack -n '__fish_spack_using_command mirror set-url' -l fetch -d 'set only the URL used for downloading'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-id -r -f -a s3_access_key_id
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-id -r -d 'ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-id-variable -r -f -a s3_access_key_id_variable
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-id-variable -r -d 'environment variable containing ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-secret -r -f -a s3_access_key_secret
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-secret -r -d 'secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-secret-variable -r -f -a s3_access_key_secret_variable
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-key-secret-variable -r -d 'environment variable containing secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-token -r -f -a s3_access_token
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-token -r -d 'access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-token-variable -r -f -a s3_access_token_variable
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-access-token-variable -r -d 'environment variable containing access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-profile -r -f -a s3_profile
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-profile -r -d 'S3 profile name to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-endpoint-url -r -f -a s3_endpoint_url
complete -c spack -n '__fish_spack_using_command mirror set-url' -l s3-endpoint-url -r -d 'endpoint URL to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-username -r -f -a oci_username
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-username -r -d 'username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-username-variable -r -f -a oci_username_variable
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-username-variable -r -d 'environment variable containing username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-password -r -f -a oci_password
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-password -r -d 'password to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-password-variable -r -f -a oci_password_variable
complete -c spack -n '__fish_spack_using_command mirror set-url' -l oci-password-variable -r -d 'environment variable containing password to use to connect to this OCI mirror'

# spack mirror set
set -g __fish_spack_optspecs_spack_mirror_set h/help push fetch type= url= autopush no-autopush unsigned signed scope= s3-access-key-id= s3-access-key-id-variable= s3-access-key-secret= s3-access-key-secret-variable= s3-access-token= s3-access-token-variable= s3-profile= s3-endpoint-url= oci-username= oci-username-variable= oci-password= oci-password-variable=
complete -c spack -n '__fish_spack_using_command_pos 0 mirror set' -f -a '(__fish_spack_mirrors)'
complete -c spack -n '__fish_spack_using_command mirror set' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror set' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror set' -l push -f -a push
complete -c spack -n '__fish_spack_using_command mirror set' -l push -d 'modify just the push connection details'
complete -c spack -n '__fish_spack_using_command mirror set' -l fetch -f -a fetch
complete -c spack -n '__fish_spack_using_command mirror set' -l fetch -d 'modify just the fetch connection details'
complete -c spack -n '__fish_spack_using_command mirror set' -l type -r -f -a 'binary source'
complete -c spack -n '__fish_spack_using_command mirror set' -l type -r -d 'specify the mirror type: for both binary and source use `--type binary --type source`'
complete -c spack -n '__fish_spack_using_command mirror set' -l url -r -f -a url
complete -c spack -n '__fish_spack_using_command mirror set' -l url -r -d 'url of mirror directory from '"'"'spack mirror create'"'"''
complete -c spack -n '__fish_spack_using_command mirror set' -l autopush -f -a autopush
complete -c spack -n '__fish_spack_using_command mirror set' -l autopush -d 'set mirror to push automatically after installation'
complete -c spack -n '__fish_spack_using_command mirror set' -l no-autopush -f -a autopush
complete -c spack -n '__fish_spack_using_command mirror set' -l no-autopush -d 'set mirror to not push automatically after installation'
complete -c spack -n '__fish_spack_using_command mirror set' -l unsigned -f -a signed
complete -c spack -n '__fish_spack_using_command mirror set' -l unsigned -d 'do not require signing and signature verification when pushing and installing from this build cache'
complete -c spack -n '__fish_spack_using_command mirror set' -l signed -f -a signed
complete -c spack -n '__fish_spack_using_command mirror set' -l signed -d 'require signing and signature verification when pushing and installing from this build cache'
complete -c spack -n '__fish_spack_using_command mirror set' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror set' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-id -r -f -a s3_access_key_id
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-id -r -d 'ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-id-variable -r -f -a s3_access_key_id_variable
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-id-variable -r -d 'environment variable containing ID string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-secret -r -f -a s3_access_key_secret
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-secret -r -d 'secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-secret-variable -r -f -a s3_access_key_secret_variable
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-key-secret-variable -r -d 'environment variable containing secret string to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-token -r -f -a s3_access_token
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-token -r -d 'access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-token-variable -r -f -a s3_access_token_variable
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-access-token-variable -r -d 'environment variable containing access token to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-profile -r -f -a s3_profile
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-profile -r -d 'S3 profile name to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-endpoint-url -r -f -a s3_endpoint_url
complete -c spack -n '__fish_spack_using_command mirror set' -l s3-endpoint-url -r -d 'endpoint URL to use to connect to this S3 mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-username -r -f -a oci_username
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-username -r -d 'username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-username-variable -r -f -a oci_username_variable
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-username-variable -r -d 'environment variable containing username to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-password -r -f -a oci_password
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-password -r -d 'password to use to connect to this OCI mirror'
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-password-variable -r -f -a oci_password_variable
complete -c spack -n '__fish_spack_using_command mirror set' -l oci-password-variable -r -d 'environment variable containing password to use to connect to this OCI mirror'

# spack mirror list
set -g __fish_spack_optspecs_spack_mirror_list h/help scope=
complete -c spack -n '__fish_spack_using_command mirror list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror list' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror list' -l scope -r -d 'configuration scope to read from'

# spack mirror ls
set -g __fish_spack_optspecs_spack_mirror_ls h/help scope=
complete -c spack -n '__fish_spack_using_command mirror ls' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command mirror ls' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command mirror ls' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command mirror ls' -l scope -r -d 'configuration scope to read from'

# spack module
set -g __fish_spack_optspecs_spack_module h/help
complete -c spack -n '__fish_spack_using_command_pos 0 module' -f -a lmod -d 'manipulate hierarchical module files'
complete -c spack -n '__fish_spack_using_command_pos 0 module' -f -a tcl -d 'manipulate non-hierarchical module files'
complete -c spack -n '__fish_spack_using_command module' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module' -s h -l help -d 'show this help message and exit'

# spack module lmod
set -g __fish_spack_optspecs_spack_module_lmod h/help n/name=
complete -c spack -n '__fish_spack_using_command_pos 0 module lmod' -f -a refresh -d 'regenerate module files'
complete -c spack -n '__fish_spack_using_command_pos 0 module lmod' -f -a find -d 'find module files for packages'
complete -c spack -n '__fish_spack_using_command_pos 0 module lmod' -f -a rm -d 'remove module files'
complete -c spack -n '__fish_spack_using_command_pos 0 module lmod' -f -a loads -d 'prompt the list of modules associated with a constraint'
complete -c spack -n '__fish_spack_using_command_pos 0 module lmod' -f -a setdefault -d 'set the default module file for a package'
complete -c spack -n '__fish_spack_using_command module lmod' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module lmod' -s n -l name -r -f -a module_set_name
complete -c spack -n '__fish_spack_using_command module lmod' -s n -l name -r -d 'named module set to use from modules configuration'

# spack module lmod refresh
set -g __fish_spack_optspecs_spack_module_lmod_refresh h/help delete-tree upstream-modules y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module lmod refresh' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module lmod refresh' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod refresh' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module lmod refresh' -l delete-tree -f -a delete_tree
complete -c spack -n '__fish_spack_using_command module lmod refresh' -l delete-tree -d 'delete the module file tree before refresh'
complete -c spack -n '__fish_spack_using_command module lmod refresh' -l upstream-modules -f -a upstream_modules
complete -c spack -n '__fish_spack_using_command module lmod refresh' -l upstream-modules -d 'generate modules for packages installed upstream'
complete -c spack -n '__fish_spack_using_command module lmod refresh' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command module lmod refresh' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack module lmod find
set -g __fish_spack_optspecs_spack_module_lmod_find h/help full-path r/dependencies
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module lmod find' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module lmod find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod find' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module lmod find' -l full-path -f -a full_path
complete -c spack -n '__fish_spack_using_command module lmod find' -l full-path -d 'display full path to module file'
complete -c spack -n '__fish_spack_using_command module lmod find' -s r -l dependencies -f -a recurse_dependencies
complete -c spack -n '__fish_spack_using_command module lmod find' -s r -l dependencies -d 'recursively traverse spec dependencies'

# spack module lmod rm
set -g __fish_spack_optspecs_spack_module_lmod_rm h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module lmod rm' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module lmod rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module lmod rm' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command module lmod rm' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack module lmod loads
set -g __fish_spack_optspecs_spack_module_lmod_loads h/help input-only p/prefix= x/exclude= r/dependencies
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module lmod loads' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module lmod loads' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod loads' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module lmod loads' -l input-only -f -a shell
complete -c spack -n '__fish_spack_using_command module lmod loads' -l input-only -d 'generate input for module command (instead of a shell script)'
complete -c spack -n '__fish_spack_using_command module lmod loads' -s p -l prefix -r -f -a prefix
complete -c spack -n '__fish_spack_using_command module lmod loads' -s p -l prefix -r -d 'prepend to module names when issuing module load commands'
complete -c spack -n '__fish_spack_using_command module lmod loads' -s x -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command module lmod loads' -s x -l exclude -r -d 'exclude package from output; may be specified multiple times'
complete -c spack -n '__fish_spack_using_command module lmod loads' -s r -l dependencies -f -a recurse_dependencies
complete -c spack -n '__fish_spack_using_command module lmod loads' -s r -l dependencies -d 'recursively traverse spec dependencies'

# spack module lmod setdefault
set -g __fish_spack_optspecs_spack_module_lmod_setdefault h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module lmod setdefault' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module lmod setdefault' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module lmod setdefault' -s h -l help -d 'show this help message and exit'

# spack module tcl
set -g __fish_spack_optspecs_spack_module_tcl h/help n/name=
complete -c spack -n '__fish_spack_using_command_pos 0 module tcl' -f -a refresh -d 'regenerate module files'
complete -c spack -n '__fish_spack_using_command_pos 0 module tcl' -f -a find -d 'find module files for packages'
complete -c spack -n '__fish_spack_using_command_pos 0 module tcl' -f -a rm -d 'remove module files'
complete -c spack -n '__fish_spack_using_command_pos 0 module tcl' -f -a loads -d 'prompt the list of modules associated with a constraint'
complete -c spack -n '__fish_spack_using_command_pos 0 module tcl' -f -a setdefault -d 'set the default module file for a package'
complete -c spack -n '__fish_spack_using_command module tcl' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module tcl' -s n -l name -r -f -a module_set_name
complete -c spack -n '__fish_spack_using_command module tcl' -s n -l name -r -d 'named module set to use from modules configuration'

# spack module tcl refresh
set -g __fish_spack_optspecs_spack_module_tcl_refresh h/help delete-tree upstream-modules y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module tcl refresh' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module tcl refresh' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl refresh' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module tcl refresh' -l delete-tree -f -a delete_tree
complete -c spack -n '__fish_spack_using_command module tcl refresh' -l delete-tree -d 'delete the module file tree before refresh'
complete -c spack -n '__fish_spack_using_command module tcl refresh' -l upstream-modules -f -a upstream_modules
complete -c spack -n '__fish_spack_using_command module tcl refresh' -l upstream-modules -d 'generate modules for packages installed upstream'
complete -c spack -n '__fish_spack_using_command module tcl refresh' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command module tcl refresh' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack module tcl find
set -g __fish_spack_optspecs_spack_module_tcl_find h/help full-path r/dependencies
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module tcl find' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module tcl find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl find' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module tcl find' -l full-path -f -a full_path
complete -c spack -n '__fish_spack_using_command module tcl find' -l full-path -d 'display full path to module file'
complete -c spack -n '__fish_spack_using_command module tcl find' -s r -l dependencies -f -a recurse_dependencies
complete -c spack -n '__fish_spack_using_command module tcl find' -s r -l dependencies -d 'recursively traverse spec dependencies'

# spack module tcl rm
set -g __fish_spack_optspecs_spack_module_tcl_rm h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module tcl rm' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module tcl rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module tcl rm' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command module tcl rm' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack module tcl loads
set -g __fish_spack_optspecs_spack_module_tcl_loads h/help input-only p/prefix= x/exclude= r/dependencies
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module tcl loads' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module tcl loads' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl loads' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command module tcl loads' -l input-only -f -a shell
complete -c spack -n '__fish_spack_using_command module tcl loads' -l input-only -d 'generate input for module command (instead of a shell script)'
complete -c spack -n '__fish_spack_using_command module tcl loads' -s p -l prefix -r -f -a prefix
complete -c spack -n '__fish_spack_using_command module tcl loads' -s p -l prefix -r -d 'prepend to module names when issuing module load commands'
complete -c spack -n '__fish_spack_using_command module tcl loads' -s x -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command module tcl loads' -s x -l exclude -r -d 'exclude package from output; may be specified multiple times'
complete -c spack -n '__fish_spack_using_command module tcl loads' -s r -l dependencies -f -a recurse_dependencies
complete -c spack -n '__fish_spack_using_command module tcl loads' -s r -l dependencies -d 'recursively traverse spec dependencies'

# spack module tcl setdefault
set -g __fish_spack_optspecs_spack_module_tcl_setdefault h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 module tcl setdefault' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command module tcl setdefault' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command module tcl setdefault' -s h -l help -d 'show this help message and exit'

# spack patch
set -g __fish_spack_optspecs_spack_patch h/help n/no-checksum f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 patch' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command patch' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command patch' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command patch' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command patch' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'
complete -c spack -n '__fish_spack_using_command patch' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command patch' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command patch' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command patch' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command patch' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command patch' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command patch' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command patch' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command patch' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command patch' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack pkg
set -g __fish_spack_optspecs_spack_pkg h/help
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a add -d 'add a package to the git stage with `git add`'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a list -d 'list packages associated with a particular spack git revision'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a diff -d 'compare packages available in two different git revisions'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a added -d 'show packages added since a commit'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a changed -d 'show packages changed since a commit'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a removed -d 'show packages removed since a commit'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a grep -d 'grep for strings in package.py files from all repositories'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a source -d 'dump source code for a package'
complete -c spack -n '__fish_spack_using_command_pos 0 pkg' -f -a hash -d 'dump canonical source code hash for a package spec'
complete -c spack -n '__fish_spack_using_command pkg' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg' -s h -l help -d 'show this help message and exit'

# spack pkg add
set -g __fish_spack_optspecs_spack_pkg_add h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 pkg add' -f -a '(__fish_spack_pkg_packages)'
complete -c spack -n '__fish_spack_using_command pkg add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg add' -s h -l help -d 'show this help message and exit'

# spack pkg list
set -g __fish_spack_optspecs_spack_pkg_list h/help
complete -c spack -n '__fish_spack_using_command_pos 0 pkg list' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command pkg list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg list' -s h -l help -d 'show this help message and exit'

# spack pkg diff
set -g __fish_spack_optspecs_spack_pkg_diff h/help
complete -c spack -n '__fish_spack_using_command_pos 0 pkg diff' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command_pos 1 pkg diff' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command pkg diff' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg diff' -s h -l help -d 'show this help message and exit'

# spack pkg added
set -g __fish_spack_optspecs_spack_pkg_added h/help
complete -c spack -n '__fish_spack_using_command_pos 0 pkg added' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command_pos 1 pkg added' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command pkg added' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg added' -s h -l help -d 'show this help message and exit'

# spack pkg changed
set -g __fish_spack_optspecs_spack_pkg_changed h/help t/type=
complete -c spack -n '__fish_spack_using_command_pos 0 pkg changed' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command_pos 1 pkg changed' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command pkg changed' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg changed' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command pkg changed' -s t -l type -r -f -a type
complete -c spack -n '__fish_spack_using_command pkg changed' -s t -l type -r -d 'types of changes to show (A: added, R: removed, C: changed); default is '"'"'C'"'"''

# spack pkg removed
set -g __fish_spack_optspecs_spack_pkg_removed h/help
complete -c spack -n '__fish_spack_using_command_pos 0 pkg removed' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command_pos 1 pkg removed' -f -a '(__fish_spack_git_rev)'
complete -c spack -n '__fish_spack_using_command pkg removed' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg removed' -s h -l help -d 'show this help message and exit'

# spack pkg grep
set -g __fish_spack_optspecs_spack_pkg_grep help

complete -c spack -n '__fish_spack_using_command pkg grep' -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg grep' -l help -d 'show this help message and exit'

# spack pkg source
set -g __fish_spack_optspecs_spack_pkg_source h/help c/canonical
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 pkg source' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command pkg source' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg source' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command pkg source' -s c -l canonical -f -a canonical
complete -c spack -n '__fish_spack_using_command pkg source' -s c -l canonical -d 'dump canonical source as used by package hash'

# spack pkg hash
set -g __fish_spack_optspecs_spack_pkg_hash h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 pkg hash' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command pkg hash' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pkg hash' -s h -l help -d 'show this help message and exit'

# spack providers
set -g __fish_spack_optspecs_spack_providers h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 providers' -f -a '(__fish_spack_providers)'
complete -c spack -n '__fish_spack_using_command providers' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command providers' -s h -l help -d 'show this help message and exit'

# spack pydoc
set -g __fish_spack_optspecs_spack_pydoc h/help

complete -c spack -n '__fish_spack_using_command pydoc' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command pydoc' -s h -l help -d 'show this help message and exit'

# spack python
set -g __fish_spack_optspecs_spack_python h/help V/version c/= u/ i/= m/= path

complete -c spack -n '__fish_spack_using_command python' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command python' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command python' -s V -l version -f -a python_version
complete -c spack -n '__fish_spack_using_command python' -s V -l version -d 'print the Python version number and exit'
complete -c spack -n '__fish_spack_using_command python' -s c -r -f -a python_command
complete -c spack -n '__fish_spack_using_command python' -s c -r -d 'command to execute'
complete -c spack -n '__fish_spack_using_command python' -s u -f -a unbuffered
complete -c spack -n '__fish_spack_using_command python' -s u -d 'for compatibility with xdist, do not use without adding -u to the interpreter'
complete -c spack -n '__fish_spack_using_command python' -s i -r -f -a 'python ipython'
complete -c spack -n '__fish_spack_using_command python' -s i -r -d 'python interpreter'
complete -c spack -n '__fish_spack_using_command python' -s m -r -f -a module
complete -c spack -n '__fish_spack_using_command python' -s m -r -d 'run library module as a script'
complete -c spack -n '__fish_spack_using_command python' -l path -f -a show_path
complete -c spack -n '__fish_spack_using_command python' -l path -d 'show path to python interpreter that spack uses'

# spack reindex
set -g __fish_spack_optspecs_spack_reindex h/help
complete -c spack -n '__fish_spack_using_command reindex' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command reindex' -s h -l help -d 'show this help message and exit'

# spack remove
set -g __fish_spack_optspecs_spack_remove h/help a/all l/list-name= f/force
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 remove' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command remove' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command remove' -s a -l all -d 'remove all specs from (clear) the environment'
complete -c spack -n '__fish_spack_using_command remove' -s l -l list-name -r -f -a list_name
complete -c spack -n '__fish_spack_using_command remove' -s l -l list-name -r -d 'name of the list to remove specs from'
complete -c spack -n '__fish_spack_using_command remove' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command remove' -s f -l force -d 'remove concretized spec (if any) immediately'

# spack rm
set -g __fish_spack_optspecs_spack_rm h/help a/all l/list-name= f/force
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 rm' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command rm' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command rm' -s a -l all -d 'remove all specs from (clear) the environment'
complete -c spack -n '__fish_spack_using_command rm' -s l -l list-name -r -f -a list_name
complete -c spack -n '__fish_spack_using_command rm' -s l -l list-name -r -d 'name of the list to remove specs from'
complete -c spack -n '__fish_spack_using_command rm' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command rm' -s f -l force -d 'remove concretized spec (if any) immediately'

# spack repo
set -g __fish_spack_optspecs_spack_repo h/help
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a create -d 'create a new package repository'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a list -d 'show registered repositories and their namespaces'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a ls -d 'show registered repositories and their namespaces'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a add -d 'add package repositories to Spack'"'"'s configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a set -d 'modify an existing repository configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a remove -d 'remove a repository from Spack'"'"'s configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a rm -d 'remove a repository from Spack'"'"'s configuration'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a migrate -d 'migrate a package repository to the latest Package API'
complete -c spack -n '__fish_spack_using_command_pos 0 repo' -f -a update -d 'update one or more package repositories'
complete -c spack -n '__fish_spack_using_command repo' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo' -s h -l help -d 'show this help message and exit'

# spack repo create
set -g __fish_spack_optspecs_spack_repo_create h/help d/subdirectory=
complete -c spack -n '__fish_spack_using_command_pos 0 repo create' -f -a '(__fish_spack_environments)'
complete -c spack -n '__fish_spack_using_command repo create' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo create' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo create' -s d -l subdirectory -r -f -a subdir
complete -c spack -n '__fish_spack_using_command repo create' -s d -l subdirectory -r -d 'subdirectory to store packages in the repository'

# spack repo list
set -g __fish_spack_optspecs_spack_repo_list h/help scope= names namespaces
complete -c spack -n '__fish_spack_using_command repo list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo list' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo list' -l scope -r -d 'configuration scope to read from'
complete -c spack -n '__fish_spack_using_command repo list' -l names -f -a names
complete -c spack -n '__fish_spack_using_command repo list' -l names -d 'show configuration names only'
complete -c spack -n '__fish_spack_using_command repo list' -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command repo list' -l namespaces -d 'show repository namespaces only'

# spack repo ls
set -g __fish_spack_optspecs_spack_repo_ls h/help scope= names namespaces
complete -c spack -n '__fish_spack_using_command repo ls' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo ls' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo ls' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo ls' -l scope -r -d 'configuration scope to read from'
complete -c spack -n '__fish_spack_using_command repo ls' -l names -f -a names
complete -c spack -n '__fish_spack_using_command repo ls' -l names -d 'show configuration names only'
complete -c spack -n '__fish_spack_using_command repo ls' -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command repo ls' -l namespaces -d 'show repository namespaces only'

# spack repo add
set -g __fish_spack_optspecs_spack_repo_add h/help name= path= scope=

complete -c spack -n '__fish_spack_using_command repo add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo add' -l name -r -f -a name
complete -c spack -n '__fish_spack_using_command repo add' -l name -r -d 'config name for the package repository, defaults to the namespace of the repository'
complete -c spack -n '__fish_spack_using_command repo add' -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command repo add' -l path -r -d 'relative path to the Spack package repository inside a git repository. Can be repeated to add multiple package repositories in case of a monorepo'
complete -c spack -n '__fish_spack_using_command repo add' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo add' -l scope -r -d 'configuration scope to modify'

# spack repo set
set -g __fish_spack_optspecs_spack_repo_set h/help destination= path= scope=

complete -c spack -n '__fish_spack_using_command repo set' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo set' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo set' -l destination -r -f -a destination
complete -c spack -n '__fish_spack_using_command repo set' -l destination -r -d 'destination to clone git repository into'
complete -c spack -n '__fish_spack_using_command repo set' -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command repo set' -l path -r -d 'relative path to the Spack package repository inside a git repository. Can be repeated to add multiple package repositories in case of a monorepo'
complete -c spack -n '__fish_spack_using_command repo set' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo set' -l scope -r -d 'configuration scope to modify'

# spack repo remove
set -g __fish_spack_optspecs_spack_repo_remove h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 repo remove' $__fish_spack_force_files -a '(__fish_spack_repos)'
complete -c spack -n '__fish_spack_using_command repo remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo remove' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo remove' -l scope -r -d 'configuration scope to modify'

# spack repo rm
set -g __fish_spack_optspecs_spack_repo_rm h/help scope=
complete -c spack -n '__fish_spack_using_command_pos 0 repo rm' $__fish_spack_force_files -a '(__fish_spack_repos)'
complete -c spack -n '__fish_spack_using_command repo rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo rm' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo rm' -l scope -r -d 'configuration scope to modify'

# spack repo migrate
set -g __fish_spack_optspecs_spack_repo_migrate h/help dry-run fix
complete -c spack -n '__fish_spack_using_command_pos 0 repo migrate' $__fish_spack_force_files -a '(__fish_spack_repos)'
complete -c spack -n '__fish_spack_using_command repo migrate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo migrate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo migrate' -l dry-run -f -a dry_run
complete -c spack -n '__fish_spack_using_command repo migrate' -l dry-run -d 'do not modify the repository, but dump a patch file'
complete -c spack -n '__fish_spack_using_command repo migrate' -l fix -f -a fix
complete -c spack -n '__fish_spack_using_command repo migrate' -l fix -d 'automatically migrate the repository to the latest Package API'

# spack repo update
set -g __fish_spack_optspecs_spack_repo_update h/help r/remote= scope= b/branch= t/tag= c/commit=

complete -c spack -n '__fish_spack_using_command repo update' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command repo update' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command repo update' -l remote -s r -r -f -a remote
complete -c spack -n '__fish_spack_using_command repo update' -l remote -s r -r -d 'name of remote to check for branches, tags, or commits'
complete -c spack -n '__fish_spack_using_command repo update' -l scope -r -f -a '_builtin defaults:base defaults system site user command_line'
complete -c spack -n '__fish_spack_using_command repo update' -l scope -r -d 'configuration scope to modify'
complete -c spack -n '__fish_spack_using_command repo update' -l branch -s b -r -f -a branch
complete -c spack -n '__fish_spack_using_command repo update' -l branch -s b -r -d 'name of a branch to change to'
complete -c spack -n '__fish_spack_using_command repo update' -l tag -s t -r -f -a tag
complete -c spack -n '__fish_spack_using_command repo update' -l tag -s t -r -d 'name of a tag to change to'
complete -c spack -n '__fish_spack_using_command repo update' -l commit -s c -r -f -a commit
complete -c spack -n '__fish_spack_using_command repo update' -l commit -s c -r -d 'name of a commit to change to'

# spack resource
set -g __fish_spack_optspecs_spack_resource h/help
complete -c spack -n '__fish_spack_using_command_pos 0 resource' -f -a list -d 'list all resources known to spack (currently just patches)'
complete -c spack -n '__fish_spack_using_command_pos 0 resource' -f -a show -d 'show a resource, identified by its checksum'
complete -c spack -n '__fish_spack_using_command resource' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command resource' -s h -l help -d 'show this help message and exit'

# spack resource list
set -g __fish_spack_optspecs_spack_resource_list h/help only-hashes
complete -c spack -n '__fish_spack_using_command resource list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command resource list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command resource list' -l only-hashes -f -a only_hashes
complete -c spack -n '__fish_spack_using_command resource list' -l only-hashes -d 'only print sha256 hashes of resources'

# spack resource show
set -g __fish_spack_optspecs_spack_resource_show h/help

complete -c spack -n '__fish_spack_using_command resource show' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command resource show' -s h -l help -d 'show this help message and exit'

# spack restage
set -g __fish_spack_optspecs_spack_restage h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 restage' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command restage' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command restage' -s h -l help -d 'show this help message and exit'

# spack solve
set -g __fish_spack_optspecs_spack_solve h/help show= timers stats l/long L/very-long N/namespaces I/install-status no-install-status y/yaml j/json format= c/cover= t/types f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 solve' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command solve' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command solve' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command solve' -l show -r -f -a show
complete -c spack -n '__fish_spack_using_command solve' -l show -r -d 'select outputs'
complete -c spack -n '__fish_spack_using_command solve' -l timers -f -a timers
complete -c spack -n '__fish_spack_using_command solve' -l timers -d 'print out timers for different solve phases'
complete -c spack -n '__fish_spack_using_command solve' -l stats -f -a stats
complete -c spack -n '__fish_spack_using_command solve' -l stats -d 'print out statistics from clingo'
complete -c spack -n '__fish_spack_using_command solve' -s l -l long -f -a long
complete -c spack -n '__fish_spack_using_command solve' -s l -l long -d 'show dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command solve' -s L -l very-long -f -a very_long
complete -c spack -n '__fish_spack_using_command solve' -s L -l very-long -d 'show full dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command solve' -s N -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command solve' -s N -l namespaces -d 'show fully qualified package names'
complete -c spack -n '__fish_spack_using_command solve' -s I -l install-status -f -a install_status
complete -c spack -n '__fish_spack_using_command solve' -s I -l install-status -d 'show install status of packages'
complete -c spack -n '__fish_spack_using_command solve' -l no-install-status -f -a install_status
complete -c spack -n '__fish_spack_using_command solve' -l no-install-status -d 'do not show install status annotations'
complete -c spack -n '__fish_spack_using_command solve' -s y -l yaml -f -a format
complete -c spack -n '__fish_spack_using_command solve' -s y -l yaml -d 'print concrete spec as YAML'
complete -c spack -n '__fish_spack_using_command solve' -s j -l json -f -a format
complete -c spack -n '__fish_spack_using_command solve' -s j -l json -d 'print concrete spec as JSON'
complete -c spack -n '__fish_spack_using_command solve' -l format -r -f -a format
complete -c spack -n '__fish_spack_using_command solve' -l format -r -d 'print concrete spec with the specified format string'
complete -c spack -n '__fish_spack_using_command solve' -s c -l cover -r -f -a 'nodes edges paths'
complete -c spack -n '__fish_spack_using_command solve' -s c -l cover -r -d 'how extensively to traverse the DAG (default: nodes)'
complete -c spack -n '__fish_spack_using_command solve' -s t -l types -f -a types
complete -c spack -n '__fish_spack_using_command solve' -s t -l types -d 'show dependency types'
complete -c spack -n '__fish_spack_using_command solve' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command solve' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command solve' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command solve' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command solve' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command solve' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command solve' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command solve' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command solve' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command solve' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack spec
set -g __fish_spack_optspecs_spack_spec h/help l/long L/very-long N/namespaces I/install-status no-install-status y/yaml j/json format= c/cover= t/types f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 spec' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command spec' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command spec' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command spec' -s l -l long -f -a long
complete -c spack -n '__fish_spack_using_command spec' -s l -l long -d 'show dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command spec' -s L -l very-long -f -a very_long
complete -c spack -n '__fish_spack_using_command spec' -s L -l very-long -d 'show full dependency hashes as well as versions'
complete -c spack -n '__fish_spack_using_command spec' -s N -l namespaces -f -a namespaces
complete -c spack -n '__fish_spack_using_command spec' -s N -l namespaces -d 'show fully qualified package names'
complete -c spack -n '__fish_spack_using_command spec' -s I -l install-status -f -a install_status
complete -c spack -n '__fish_spack_using_command spec' -s I -l install-status -d 'show install status of packages'
complete -c spack -n '__fish_spack_using_command spec' -l no-install-status -f -a install_status
complete -c spack -n '__fish_spack_using_command spec' -l no-install-status -d 'do not show install status annotations'
complete -c spack -n '__fish_spack_using_command spec' -s y -l yaml -f -a format
complete -c spack -n '__fish_spack_using_command spec' -s y -l yaml -d 'print concrete spec as YAML'
complete -c spack -n '__fish_spack_using_command spec' -s j -l json -f -a format
complete -c spack -n '__fish_spack_using_command spec' -s j -l json -d 'print concrete spec as JSON'
complete -c spack -n '__fish_spack_using_command spec' -l format -r -f -a format
complete -c spack -n '__fish_spack_using_command spec' -l format -r -d 'print concrete spec with the specified format string'
complete -c spack -n '__fish_spack_using_command spec' -s c -l cover -r -f -a 'nodes edges paths'
complete -c spack -n '__fish_spack_using_command spec' -s c -l cover -r -d 'how extensively to traverse the DAG (default: nodes)'
complete -c spack -n '__fish_spack_using_command spec' -s t -l types -f -a types
complete -c spack -n '__fish_spack_using_command spec' -s t -l types -d 'show dependency types'
complete -c spack -n '__fish_spack_using_command spec' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command spec' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command spec' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command spec' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command spec' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command spec' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command spec' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command spec' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command spec' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command spec' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack stage
set -g __fish_spack_optspecs_spack_stage h/help n/no-checksum p/path= e/exclude= s/skip-installed f/force U/fresh reuse fresh-roots deprecated
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 stage' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command stage' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command stage' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command stage' -s n -l no-checksum -f -a no_checksum
complete -c spack -n '__fish_spack_using_command stage' -s n -l no-checksum -d 'do not use checksums to verify downloaded files (unsafe)'
complete -c spack -n '__fish_spack_using_command stage' -s p -l path -r -f -a path
complete -c spack -n '__fish_spack_using_command stage' -s p -l path -r -d 'path to stage package, does not add to spack tree'
complete -c spack -n '__fish_spack_using_command stage' -s e -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command stage' -s e -l exclude -r -d 'exclude packages that satisfy the specified specs'
complete -c spack -n '__fish_spack_using_command stage' -s s -l skip-installed -f -a skip_installed
complete -c spack -n '__fish_spack_using_command stage' -s s -l skip-installed -d 'dont restage already installed specs'
complete -c spack -n '__fish_spack_using_command stage' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command stage' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command stage' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command stage' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command stage' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command stage' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command stage' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command stage' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command stage' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command stage' -l deprecated -d 'allow concretizer to select deprecated versions'

# spack style
set -g __fish_spack_optspecs_spack_style h/help b/base= a/all r/root-relative U/no-untracked f/fix root= t/tool= s/skip= spec-strings

complete -c spack -n '__fish_spack_using_command style' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command style' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command style' -s b -l base -r -f -a base
complete -c spack -n '__fish_spack_using_command style' -s b -l base -r -d 'branch to compare against to determine changed files (default: develop)'
complete -c spack -n '__fish_spack_using_command style' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command style' -s a -l all -d 'check all files, not just changed files'
complete -c spack -n '__fish_spack_using_command style' -s r -l root-relative -f -a root_relative
complete -c spack -n '__fish_spack_using_command style' -s r -l root-relative -d 'print root-relative paths (default: cwd-relative)'
complete -c spack -n '__fish_spack_using_command style' -s U -l no-untracked -f -a untracked
complete -c spack -n '__fish_spack_using_command style' -s U -l no-untracked -d 'exclude untracked files from checks'
complete -c spack -n '__fish_spack_using_command style' -s f -l fix -f -a fix
complete -c spack -n '__fish_spack_using_command style' -s f -l fix -d 'format automatically if possible (e.g., with isort, black)'
complete -c spack -n '__fish_spack_using_command style' -l root -r -f -a root
complete -c spack -n '__fish_spack_using_command style' -l root -r -d 'style check a different spack instance'
complete -c spack -n '__fish_spack_using_command style' -s t -l tool -r -f -a tool
complete -c spack -n '__fish_spack_using_command style' -s t -l tool -r -d 'specify which tools to run (default: import, isort, black, flake8, mypy)'
complete -c spack -n '__fish_spack_using_command style' -s s -l skip -r -f -a skip
complete -c spack -n '__fish_spack_using_command style' -s s -l skip -r -d 'specify tools to skip (choose from import, isort, black, flake8, mypy)'
complete -c spack -n '__fish_spack_using_command style' -l spec-strings -f -a spec_strings
complete -c spack -n '__fish_spack_using_command style' -l spec-strings -d 'upgrade spec strings in Python, JSON and YAML files for compatibility with Spack v1.0 and v0.x. Example: spack style --spec-strings $(git ls-files). Note: must be used only on specs from spack v0.X.'

# spack tags
set -g __fish_spack_optspecs_spack_tags h/help i/installed a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 tags' -f -a '(__fish_spack_tags)'
complete -c spack -n '__fish_spack_using_command tags' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command tags' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command tags' -s i -l installed -f -a installed
complete -c spack -n '__fish_spack_using_command tags' -s i -l installed -d 'show information for installed packages only'
complete -c spack -n '__fish_spack_using_command tags' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command tags' -s a -l all -d 'show packages for all available tags'

# spack test
set -g __fish_spack_optspecs_spack_test h/help
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a run -d 'run tests for the specified installed packages'
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a list -d 'list installed packages with available tests'
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a find -d 'find tests that are running or have available results'
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a status -d 'get the current status for the specified Spack test suite(s)'
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a results -d 'get the results from Spack test suite(s) (default all)'
complete -c spack -n '__fish_spack_using_command_pos 0 test' -f -a remove -d 'remove results from Spack test suite(s) (default all)'
complete -c spack -n '__fish_spack_using_command test' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test' -s h -l help -d 'show this help message and exit'

# spack test run
set -g __fish_spack_optspecs_spack_test_run h/help alias= fail-fast fail-first externals x/explicit keep-stage log-format= log-file= cdash-upload-url= cdash-build= cdash-site= cdash-track= cdash-buildstamp= help-cdash timeout= clean dirty
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test run' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command test run' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test run' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command test run' -l alias -r -f -a alias
complete -c spack -n '__fish_spack_using_command test run' -l alias -r -d 'provide an alias for this test-suite for subsequent access'
complete -c spack -n '__fish_spack_using_command test run' -l fail-fast -f -a fail_fast
complete -c spack -n '__fish_spack_using_command test run' -l fail-fast -d 'stop tests for each package after the first failure'
complete -c spack -n '__fish_spack_using_command test run' -l fail-first -f -a fail_first
complete -c spack -n '__fish_spack_using_command test run' -l fail-first -d 'stop after the first failed package'
complete -c spack -n '__fish_spack_using_command test run' -l externals -f -a externals
complete -c spack -n '__fish_spack_using_command test run' -l externals -d 'test packages that are externally installed'
complete -c spack -n '__fish_spack_using_command test run' -s x -l explicit -f -a explicit
complete -c spack -n '__fish_spack_using_command test run' -s x -l explicit -d 'only test packages that are explicitly installed'
complete -c spack -n '__fish_spack_using_command test run' -l keep-stage -f -a keep_stage
complete -c spack -n '__fish_spack_using_command test run' -l keep-stage -d 'keep testing directory for debugging'
complete -c spack -n '__fish_spack_using_command test run' -l log-format -r -f -a 'junit cdash'
complete -c spack -n '__fish_spack_using_command test run' -l log-format -r -d 'format to be used for log files'
complete -c spack -n '__fish_spack_using_command test run' -l log-file -r -f -a log_file
complete -c spack -n '__fish_spack_using_command test run' -l log-file -r -d 'filename for the log file'
complete -c spack -n '__fish_spack_using_command test run' -l cdash-upload-url -r -f -a cdash_upload_url
complete -c spack -n '__fish_spack_using_command test run' -l cdash-build -r -f -a cdash_build
complete -c spack -n '__fish_spack_using_command test run' -l cdash-site -r -f -a cdash_site
complete -c spack -n '__fish_spack_using_command test run' -l cdash-track -r -f -a cdash_track
complete -c spack -n '__fish_spack_using_command test run' -l cdash-buildstamp -r -f -a cdash_buildstamp
complete -c spack -n '__fish_spack_using_command test run' -l help-cdash -f -a help_cdash
complete -c spack -n '__fish_spack_using_command test run' -l help-cdash -d 'show usage instructions for CDash reporting'
complete -c spack -n '__fish_spack_using_command test run' -l timeout -r -f -a timeout
complete -c spack -n '__fish_spack_using_command test run' -l timeout -r -d 'maximum time (in seconds) that tests are allowed to run'
complete -c spack -n '__fish_spack_using_command test run' -l clean -f -a dirty
complete -c spack -n '__fish_spack_using_command test run' -l clean -d 'unset harmful variables in the build environment (default)'
complete -c spack -n '__fish_spack_using_command test run' -l dirty -f -a dirty
complete -c spack -n '__fish_spack_using_command test run' -l dirty -d 'preserve user environment in spack'"'"'s build environment (danger!)'

# spack test list
set -g __fish_spack_optspecs_spack_test_list h/help a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test list' -f -a '(__fish_spack_tags)'
complete -c spack -n '__fish_spack_using_command test list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command test list' -s a -l all -f -a list_all
complete -c spack -n '__fish_spack_using_command test list' -s a -l all -d 'list all packages with tests (not just installed)'

# spack test find
set -g __fish_spack_optspecs_spack_test_find h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test find' -f -a '(__fish_spack_tests)'
complete -c spack -n '__fish_spack_using_command test find' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test find' -s h -l help -d 'show this help message and exit'

# spack test status
set -g __fish_spack_optspecs_spack_test_status h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test status' -f -a '(__fish_spack_tests)'
complete -c spack -n '__fish_spack_using_command test status' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test status' -s h -l help -d 'show this help message and exit'

# spack test results
set -g __fish_spack_optspecs_spack_test_results h/help l/logs f/failed
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test results' -f -a '(__fish_spack_tests)'
complete -c spack -n '__fish_spack_using_command test results' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test results' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command test results' -s l -l logs -f -a logs
complete -c spack -n '__fish_spack_using_command test results' -s l -l logs -d 'print the test log for each matching package'
complete -c spack -n '__fish_spack_using_command test results' -s f -l failed -f -a failed
complete -c spack -n '__fish_spack_using_command test results' -s f -l failed -d 'only show results for failed tests of matching packages'

# spack test remove
set -g __fish_spack_optspecs_spack_test_remove h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test remove' -f -a '(__fish_spack_tests)'
complete -c spack -n '__fish_spack_using_command test remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command test remove' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command test remove' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack test-env
set -g __fish_spack_optspecs_spack_test_env h/help clean dirty f/force U/fresh reuse fresh-roots deprecated dump= pickle=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 test-env' -f -a '(__fish_spack_build_env_spec)'
complete -c spack -n '__fish_spack_using_command test-env' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command test-env' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command test-env' -l clean -f -a dirty
complete -c spack -n '__fish_spack_using_command test-env' -l clean -d 'unset harmful variables in the build environment (default)'
complete -c spack -n '__fish_spack_using_command test-env' -l dirty -f -a dirty
complete -c spack -n '__fish_spack_using_command test-env' -l dirty -d 'preserve user environment in spack'"'"'s build environment (danger!)'
complete -c spack -n '__fish_spack_using_command test-env' -s f -l force -f -a concretizer_force
complete -c spack -n '__fish_spack_using_command test-env' -s f -l force -d 'allow changes to concretized specs in spack.lock (in an env)'
complete -c spack -n '__fish_spack_using_command test-env' -s U -l fresh -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command test-env' -s U -l fresh -d 'do not reuse installed deps; build newest configuration'
complete -c spack -n '__fish_spack_using_command test-env' -l reuse -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command test-env' -l reuse -d 'reuse installed packages/buildcaches when possible'
complete -c spack -n '__fish_spack_using_command test-env' -l fresh-roots -l reuse-deps -f -a concretizer_reuse
complete -c spack -n '__fish_spack_using_command test-env' -l fresh-roots -l reuse-deps -d 'concretize with fresh roots and reused dependencies'
complete -c spack -n '__fish_spack_using_command test-env' -l deprecated -f -a config_deprecated
complete -c spack -n '__fish_spack_using_command test-env' -l deprecated -d 'allow concretizer to select deprecated versions'
complete -c spack -n '__fish_spack_using_command test-env' -l dump -r -f -a dump
complete -c spack -n '__fish_spack_using_command test-env' -l dump -r -d 'dump a source-able environment to FILE'
complete -c spack -n '__fish_spack_using_command test-env' -l pickle -r -f -a pickle
complete -c spack -n '__fish_spack_using_command test-env' -l pickle -r -d 'dump a pickled source-able environment to FILE'

# spack tutorial
set -g __fish_spack_optspecs_spack_tutorial h/help y/yes-to-all
complete -c spack -n '__fish_spack_using_command tutorial' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command tutorial' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command tutorial' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command tutorial' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'

# spack undevelop
set -g __fish_spack_optspecs_spack_undevelop h/help a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 undevelop' -f -k -a '(__fish_spack_specs_or_id)'
complete -c spack -n '__fish_spack_using_command undevelop' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command undevelop' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command undevelop' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command undevelop' -s a -l all -d 'remove all specs from (clear) the environment'

# spack uninstall
set -g __fish_spack_optspecs_spack_uninstall h/help f/force remove R/dependents y/yes-to-all a/all origin=
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 uninstall' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command uninstall' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command uninstall' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command uninstall' -s f -l force -f -a force
complete -c spack -n '__fish_spack_using_command uninstall' -s f -l force -d 'remove regardless of whether other packages or environments depend on this one'
complete -c spack -n '__fish_spack_using_command uninstall' -l remove -f -a remove
complete -c spack -n '__fish_spack_using_command uninstall' -l remove -d 'if in an environment, then the spec should also be removed from the environment description'
complete -c spack -n '__fish_spack_using_command uninstall' -s R -l dependents -f -a dependents
complete -c spack -n '__fish_spack_using_command uninstall' -s R -l dependents -d 'also uninstall any packages that depend on the ones given via command line'
complete -c spack -n '__fish_spack_using_command uninstall' -s y -l yes-to-all -f -a yes_to_all
complete -c spack -n '__fish_spack_using_command uninstall' -s y -l yes-to-all -d 'assume "yes" is the answer to every confirmation request'
complete -c spack -n '__fish_spack_using_command uninstall' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command uninstall' -s a -l all -d 'remove ALL installed packages that match each supplied spec'
complete -c spack -n '__fish_spack_using_command uninstall' -l origin -r -f -a origin
complete -c spack -n '__fish_spack_using_command uninstall' -l origin -r -d 'only remove DB records with the specified origin'

# spack unit-test
set -g __fish_spack_optspecs_spack_unit_test h/help H/pytest-help n/numprocesses= l/list L/list-long N/list-names extension= s/ k/= showlocals
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 unit-test' -f -a '(__fish_spack_unit_tests)'
complete -c spack -n '__fish_spack_using_command unit-test' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command unit-test' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command unit-test' -s H -l pytest-help -f -a pytest_help
complete -c spack -n '__fish_spack_using_command unit-test' -s H -l pytest-help -d 'show full pytest help, with advanced options'
complete -c spack -n '__fish_spack_using_command unit-test' -s n -l numprocesses -r -f -a numprocesses
complete -c spack -n '__fish_spack_using_command unit-test' -s n -l numprocesses -r -d 'run tests in parallel up to this wide, default 1 for sequential'
complete -c spack -n '__fish_spack_using_command unit-test' -s l -l list -f -a list
complete -c spack -n '__fish_spack_using_command unit-test' -s l -l list -d 'list test filenames'
complete -c spack -n '__fish_spack_using_command unit-test' -s L -l list-long -f -a list
complete -c spack -n '__fish_spack_using_command unit-test' -s L -l list-long -d 'list all test functions'
complete -c spack -n '__fish_spack_using_command unit-test' -s N -l list-names -f -a list
complete -c spack -n '__fish_spack_using_command unit-test' -s N -l list-names -d 'list full names of all tests'
complete -c spack -n '__fish_spack_using_command unit-test' -l extension -r -f -a extension
complete -c spack -n '__fish_spack_using_command unit-test' -l extension -r -d 'run test for a given spack extension'
complete -c spack -n '__fish_spack_using_command unit-test' -s s -f -a parsed_args
complete -c spack -n '__fish_spack_using_command unit-test' -s s -d 'print output while tests run (disable capture)'
complete -c spack -n '__fish_spack_using_command unit-test' -s k -r -f -a expression
complete -c spack -n '__fish_spack_using_command unit-test' -s k -r -d 'filter tests by keyword (can also use w/list options)'
complete -c spack -n '__fish_spack_using_command unit-test' -l showlocals -f -a parsed_args
complete -c spack -n '__fish_spack_using_command unit-test' -l showlocals -d 'show local variable values in tracebacks'

# spack unload
set -g __fish_spack_optspecs_spack_unload h/help sh csh fish bat pwsh a/all
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 unload' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command unload' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command unload' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command unload' -l sh -f -a shell
complete -c spack -n '__fish_spack_using_command unload' -l sh -d 'print sh commands to activate the environment'
complete -c spack -n '__fish_spack_using_command unload' -l csh -f -a shell
complete -c spack -n '__fish_spack_using_command unload' -l csh -d 'print csh commands to activate the environment'
complete -c spack -n '__fish_spack_using_command unload' -l fish -f -a shell
complete -c spack -n '__fish_spack_using_command unload' -l fish -d 'print fish commands to load the package'
complete -c spack -n '__fish_spack_using_command unload' -l bat -f -a shell
complete -c spack -n '__fish_spack_using_command unload' -l bat -d 'print bat commands to load the package'
complete -c spack -n '__fish_spack_using_command unload' -l pwsh -f -a shell
complete -c spack -n '__fish_spack_using_command unload' -l pwsh -d 'print pwsh commands to load the package'
complete -c spack -n '__fish_spack_using_command unload' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command unload' -s a -l all -d 'unload all loaded Spack packages'

# spack url
set -g __fish_spack_optspecs_spack_url h/help
complete -c spack -n '__fish_spack_using_command_pos 0 url' -f -a parse -d 'attempt to parse a url'
complete -c spack -n '__fish_spack_using_command_pos 0 url' -f -a list -d 'list urls in all packages'
complete -c spack -n '__fish_spack_using_command_pos 0 url' -f -a summary -d 'print a summary of how well we are parsing package urls'
complete -c spack -n '__fish_spack_using_command_pos 0 url' -f -a stats -d 'print statistics on versions and checksums for all packages'
complete -c spack -n '__fish_spack_using_command url' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command url' -s h -l help -d 'show this help message and exit'

# spack url parse
set -g __fish_spack_optspecs_spack_url_parse h/help s/spider

complete -c spack -n '__fish_spack_using_command url parse' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command url parse' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command url parse' -s s -l spider -f -a spider
complete -c spack -n '__fish_spack_using_command url parse' -s s -l spider -d 'spider the source page for versions'

# spack url list
set -g __fish_spack_optspecs_spack_url_list h/help c/color e/extrapolation n/incorrect-name N/correct-name v/incorrect-version V/correct-version
complete -c spack -n '__fish_spack_using_command url list' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command url list' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command url list' -s c -l color -f -a color
complete -c spack -n '__fish_spack_using_command url list' -s c -l color -d 'color the parsed version and name in the urls shown (versions will be cyan, name red)'
complete -c spack -n '__fish_spack_using_command url list' -s e -l extrapolation -f -a extrapolation
complete -c spack -n '__fish_spack_using_command url list' -s e -l extrapolation -d 'color the versions used for extrapolation as well (additional versions will be green, names magenta)'
complete -c spack -n '__fish_spack_using_command url list' -s n -l incorrect-name -f -a incorrect_name
complete -c spack -n '__fish_spack_using_command url list' -s n -l incorrect-name -d 'only list urls for which the name was incorrectly parsed'
complete -c spack -n '__fish_spack_using_command url list' -s N -l correct-name -f -a correct_name
complete -c spack -n '__fish_spack_using_command url list' -s N -l correct-name -d 'only list urls for which the name was correctly parsed'
complete -c spack -n '__fish_spack_using_command url list' -s v -l incorrect-version -f -a incorrect_version
complete -c spack -n '__fish_spack_using_command url list' -s v -l incorrect-version -d 'only list urls for which the version was incorrectly parsed'
complete -c spack -n '__fish_spack_using_command url list' -s V -l correct-version -f -a correct_version
complete -c spack -n '__fish_spack_using_command url list' -s V -l correct-version -d 'only list urls for which the version was correctly parsed'

# spack url summary
set -g __fish_spack_optspecs_spack_url_summary h/help
complete -c spack -n '__fish_spack_using_command url summary' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command url summary' -s h -l help -d 'show this help message and exit'

# spack url stats
set -g __fish_spack_optspecs_spack_url_stats h/help show-issues
complete -c spack -n '__fish_spack_using_command url stats' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command url stats' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command url stats' -l show-issues -f -a show_issues
complete -c spack -n '__fish_spack_using_command url stats' -l show-issues -d 'show packages with issues (md5 hashes, http urls)'

# spack verify
set -g __fish_spack_optspecs_spack_verify h/help
complete -c spack -n '__fish_spack_using_command_pos 0 verify' -f -a manifest -d 'verify that install directories have not been modified since installation'
complete -c spack -n '__fish_spack_using_command_pos 0 verify' -f -a libraries -d 'verify that shared libraries of install packages can be located in rpaths (Linux only)'
complete -c spack -n '__fish_spack_using_command verify' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command verify' -s h -l help -d 'show this help message and exit'

# spack verify manifest
set -g __fish_spack_optspecs_spack_verify_manifest h/help l/local j/json a/all s/specs f/files
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 verify manifest' $__fish_spack_force_files -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command verify manifest' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command verify manifest' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command verify manifest' -s l -l local -f -a local
complete -c spack -n '__fish_spack_using_command verify manifest' -s l -l local -d 'verify only locally installed packages'
complete -c spack -n '__fish_spack_using_command verify manifest' -s j -l json -f -a json
complete -c spack -n '__fish_spack_using_command verify manifest' -s j -l json -d 'ouptut json-formatted errors'
complete -c spack -n '__fish_spack_using_command verify manifest' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command verify manifest' -s a -l all -d 'verify all packages'
complete -c spack -n '__fish_spack_using_command verify manifest' -s s -l specs -f -a type
complete -c spack -n '__fish_spack_using_command verify manifest' -s s -l specs -d 'treat entries as specs (default)'
complete -c spack -n '__fish_spack_using_command verify manifest' -s f -l files -f -a type
complete -c spack -n '__fish_spack_using_command verify manifest' -s f -l files -d 'treat entries as absolute filenames'

# spack verify libraries
set -g __fish_spack_optspecs_spack_verify_libraries h/help
complete -c spack -n '__fish_spack_using_command_pos_remainder 0 verify libraries' -f -a '(__fish_spack_installed_specs)'
complete -c spack -n '__fish_spack_using_command verify libraries' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command verify libraries' -s h -l help -d 'show this help message and exit'

# spack versions
set -g __fish_spack_optspecs_spack_versions h/help s/safe r/remote n/new j/jobs=
complete -c spack -n '__fish_spack_using_command_pos 0 versions' -f -a '(__fish_spack_packages)'
complete -c spack -n '__fish_spack_using_command versions' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command versions' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command versions' -s s -l safe -f -a safe
complete -c spack -n '__fish_spack_using_command versions' -s s -l safe -d 'only list safe versions of the package'
complete -c spack -n '__fish_spack_using_command versions' -s r -l remote -f -a remote
complete -c spack -n '__fish_spack_using_command versions' -s r -l remote -d 'only list remote versions of the package'
complete -c spack -n '__fish_spack_using_command versions' -s n -l new -f -a new
complete -c spack -n '__fish_spack_using_command versions' -s n -l new -d 'only list remote versions newer than the latest checksummed version'
complete -c spack -n '__fish_spack_using_command versions' -s j -l jobs -r -f -a jobs
complete -c spack -n '__fish_spack_using_command versions' -s j -l jobs -r -d 'explicitly set number of parallel jobs'

# spack view
set -g __fish_spack_optspecs_spack_view h/help v/verbose e/exclude= d/dependencies=
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a symlink -d 'add package files to a filesystem view via symbolic links'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a add -d 'add package files to a filesystem view via symbolic links'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a soft -d 'add package files to a filesystem view via symbolic links'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a hardlink -d 'add packages files to a filesystem view via hard links'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a hard -d 'add packages files to a filesystem view via hard links'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a copy -d 'add package files to a filesystem view via copy/relocate'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a relocate -d 'add package files to a filesystem view via copy/relocate'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a remove -d 'remove packages from a filesystem view'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a rm -d 'remove packages from a filesystem view'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a statlink -d 'check status of packages in a filesystem view'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a status -d 'check status of packages in a filesystem view'
complete -c spack -n '__fish_spack_using_command_pos 0 view' -f -a check -d 'check status of packages in a filesystem view'
complete -c spack -n '__fish_spack_using_command view' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view' -s v -l verbose -f -a verbose
complete -c spack -n '__fish_spack_using_command view' -s v -l verbose -d 'if not verbose only warnings/errors will be printed'
complete -c spack -n '__fish_spack_using_command view' -s e -l exclude -r -f -a exclude
complete -c spack -n '__fish_spack_using_command view' -s e -l exclude -r -d 'exclude packages with names matching the given regex pattern'
complete -c spack -n '__fish_spack_using_command view' -s d -l dependencies -r -f -a 'true false yes no'
complete -c spack -n '__fish_spack_using_command view' -s d -l dependencies -r -d 'link/remove/list dependencies'

# spack view symlink
set -g __fish_spack_optspecs_spack_view_symlink h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view symlink' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view symlink' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view symlink' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view symlink' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view symlink' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view symlink' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view symlink' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view add
set -g __fish_spack_optspecs_spack_view_add h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view add' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view add' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view add' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view add' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view add' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view add' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view add' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view soft
set -g __fish_spack_optspecs_spack_view_soft h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view soft' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view soft' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view soft' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view soft' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view soft' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view soft' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view soft' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view hardlink
set -g __fish_spack_optspecs_spack_view_hardlink h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view hardlink' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view hardlink' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view hardlink' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view hardlink' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view hardlink' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view hardlink' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view hardlink' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view hard
set -g __fish_spack_optspecs_spack_view_hard h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view hard' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view hard' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view hard' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view hard' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view hard' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view hard' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view hard' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view copy
set -g __fish_spack_optspecs_spack_view_copy h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view copy' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view copy' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view copy' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view copy' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view copy' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view copy' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view copy' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view relocate
set -g __fish_spack_optspecs_spack_view_relocate h/help projection-file= i/ignore-conflicts
complete -c spack -n '__fish_spack_using_command_pos 0 view relocate' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view relocate' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view relocate' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view relocate' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view relocate' -l projection-file -r -f -a projection_file
complete -c spack -n '__fish_spack_using_command view relocate' -l projection-file -r -d 'initialize view using projections from file'
complete -c spack -n '__fish_spack_using_command view relocate' -s i -l ignore-conflicts -f -a ignore_conflicts

# spack view remove
set -g __fish_spack_optspecs_spack_view_remove h/help no-remove-dependents a/all
complete -c spack -n '__fish_spack_using_command_pos 0 view remove' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view remove' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view remove' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view remove' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view remove' -l no-remove-dependents -f -a no_remove_dependents
complete -c spack -n '__fish_spack_using_command view remove' -l no-remove-dependents -d 'do not remove dependents of specified specs'
complete -c spack -n '__fish_spack_using_command view remove' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command view remove' -s a -l all -d 'act on all specs in view'

# spack view rm
set -g __fish_spack_optspecs_spack_view_rm h/help no-remove-dependents a/all
complete -c spack -n '__fish_spack_using_command_pos 0 view rm' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view rm' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view rm' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view rm' -s h -l help -d 'show this help message and exit'
complete -c spack -n '__fish_spack_using_command view rm' -l no-remove-dependents -f -a no_remove_dependents
complete -c spack -n '__fish_spack_using_command view rm' -l no-remove-dependents -d 'do not remove dependents of specified specs'
complete -c spack -n '__fish_spack_using_command view rm' -s a -l all -f -a all
complete -c spack -n '__fish_spack_using_command view rm' -s a -l all -d 'act on all specs in view'

# spack view statlink
set -g __fish_spack_optspecs_spack_view_statlink h/help
complete -c spack -n '__fish_spack_using_command_pos 0 view statlink' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view statlink' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view statlink' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view statlink' -s h -l help -d 'show this help message and exit'

# spack view status
set -g __fish_spack_optspecs_spack_view_status h/help
complete -c spack -n '__fish_spack_using_command_pos 0 view status' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view status' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view status' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view status' -s h -l help -d 'show this help message and exit'

# spack view check
set -g __fish_spack_optspecs_spack_view_check h/help
complete -c spack -n '__fish_spack_using_command_pos 0 view check' -f -a '(__fish_complete_directories)'
complete -c spack -n '__fish_spack_using_command_pos_remainder 1 view check' -f -k -a '(__fish_spack_specs)'
complete -c spack -n '__fish_spack_using_command view check' -s h -l help -f -a help
complete -c spack -n '__fish_spack_using_command view check' -s h -l help -d 'show this help message and exit'
