/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.shared;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import javafx.collections.ObservableList;
import org.jabref.logic.bibtex.FieldPreferences;
import org.jabref.logic.citationkeypattern.GlobalCitationKeyPatterns;
import org.jabref.logic.exporter.BibDatabaseWriter;
import org.jabref.logic.exporter.MetaDataSerializer;
import org.jabref.logic.importer.ParseException;
import org.jabref.logic.importer.util.MetaDataParser;
import org.jabref.logic.shared.DBMSProcessor;
import org.jabref.logic.shared.DatabaseConnection;
import org.jabref.logic.shared.DatabaseConnectionProperties;
import org.jabref.logic.shared.DatabaseNotSupportedException;
import org.jabref.logic.shared.DatabaseSynchronizer;
import org.jabref.logic.shared.event.ConnectionLostEvent;
import org.jabref.logic.shared.event.SharedEntriesNotPresentEvent;
import org.jabref.logic.shared.event.UpdateRefusedEvent;
import org.jabref.logic.shared.exception.OfflineLockException;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.event.EntriesAddedEvent;
import org.jabref.model.database.event.EntriesRemovedEvent;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.event.EntriesEvent;
import org.jabref.model.entry.event.EntriesEventSource;
import org.jabref.model.entry.event.FieldChangedEvent;
import org.jabref.model.entry.field.Field;
import org.jabref.model.metadata.MetaData;
import org.jabref.model.metadata.event.MetaDataChangedEvent;
import org.jabref.model.util.FileUpdateMonitor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DBMSSynchronizer
implements DatabaseSynchronizer {
    private static final Logger LOGGER = LoggerFactory.getLogger(DBMSSynchronizer.class);
    private DBMSProcessor dbmsProcessor;
    private String dbName;
    private final BibDatabaseContext bibDatabaseContext;
    private MetaData metaData;
    private final BibDatabase bibDatabase;
    private final EventBus eventBus;
    private Connection currentConnection;
    private final Character keywordSeparator;
    private final GlobalCitationKeyPatterns globalCiteKeyPattern;
    private final FieldPreferences fieldPreferences;
    private final FileUpdateMonitor fileMonitor;
    private Optional<BibEntry> lastEntryChanged;

    public DBMSSynchronizer(BibDatabaseContext bibDatabaseContext, Character keywordSeparator, FieldPreferences fieldPreferences, GlobalCitationKeyPatterns globalCiteKeyPattern, FileUpdateMonitor fileMonitor) {
        this.bibDatabaseContext = Objects.requireNonNull(bibDatabaseContext);
        this.bibDatabase = bibDatabaseContext.getDatabase();
        this.metaData = bibDatabaseContext.getMetaData();
        this.fieldPreferences = fieldPreferences;
        this.fileMonitor = fileMonitor;
        this.eventBus = new EventBus();
        this.keywordSeparator = keywordSeparator;
        this.globalCiteKeyPattern = Objects.requireNonNull(globalCiteKeyPattern);
        this.lastEntryChanged = Optional.empty();
    }

    @Subscribe
    public void listen(EntriesAddedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.synchronizeLocalMetaData();
            this.pullWithLastEntry();
            this.synchronizeLocalDatabase();
            this.dbmsProcessor.insertEntries(event.getBibEntries());
            this.lastEntryChanged = Optional.empty();
        }
    }

    @Subscribe
    public void listen(FieldChangedEvent event) {
        BibEntry bibEntry = event.getBibEntry();
        if (this.isPresentLocalBibEntry(bibEntry) && this.isEventSourceAccepted(event) && this.checkCurrentConnection() && !event.isFilteredOut()) {
            this.synchronizeLocalMetaData();
            this.pullWithLastEntry();
            this.synchronizeSharedEntry(bibEntry);
            this.synchronizeLocalDatabase();
        } else {
            this.lastEntryChanged = Optional.of(bibEntry);
        }
    }

    @Subscribe
    public void listen(EntriesRemovedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.synchronizeLocalMetaData();
            this.pullWithLastEntry();
            this.dbmsProcessor.removeEntries(event.getBibEntries());
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(MetaDataChangedEvent event) {
        if (this.checkCurrentConnection()) {
            this.synchronizeSharedMetaData(event.getMetaData(), this.globalCiteKeyPattern);
            this.synchronizeLocalDatabase();
            this.applyMetaData();
            this.dbmsProcessor.notifyClients();
        }
    }

    public void initializeDatabases() throws DatabaseNotSupportedException {
        try {
            if (!this.dbmsProcessor.checkBaseIntegrity()) {
                LOGGER.info("Integrity check failed. Fixing...");
                if (this.dbmsProcessor.databaseIsAtMostJabRef35()) {
                    throw new DatabaseNotSupportedException();
                }
                this.dbmsProcessor.setupSharedDatabase();
            }
        }
        catch (SQLException e) {
            LOGGER.error("Could not check intergrity", (Throwable)e);
            throw new IllegalStateException(e);
        }
        this.dbmsProcessor.startNotificationListener(this);
        this.synchronizeLocalMetaData();
        this.synchronizeLocalDatabase();
    }

    @Override
    public void synchronizeLocalDatabase() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        ObservableList<BibEntry> localEntries = this.bibDatabase.getEntries();
        Map<Integer, Integer> idVersionMap = this.dbmsProcessor.getSharedIDVersionMapping();
        this.removeNotSharedEntries((List<BibEntry>)localEntries, idVersionMap.keySet());
        ArrayList<Integer> entriesToInsertIntoLocalDatabase = new ArrayList<Integer>();
        for (Map.Entry<Integer, Integer> idVersionEntry : idVersionMap.entrySet()) {
            boolean remoteEntryMatchingOneLocalEntryFound = false;
            for (BibEntry localEntry : localEntries) {
                Optional<BibEntry> sharedEntry;
                if (!idVersionEntry.getKey().equals(localEntry.getSharedBibEntryData().getSharedID())) continue;
                remoteEntryMatchingOneLocalEntryFound = true;
                if (idVersionEntry.getValue() <= localEntry.getSharedBibEntryData().getVersion() || !(sharedEntry = this.dbmsProcessor.getSharedEntry(idVersionEntry.getKey())).isPresent()) continue;
                localEntry.setType(sharedEntry.get().getType(), EntriesEventSource.SHARED);
                localEntry.getSharedBibEntryData().setVersion(sharedEntry.get().getSharedBibEntryData().getVersion());
                sharedEntry.get().getFieldMap().forEach((field, value) -> localEntry.setField((Field)field, (String)value, EntriesEventSource.SHARED));
                localEntry.getFields().stream().filter(field -> !((BibEntry)sharedEntry.get()).hasField((Field)field)).forEach(field -> localEntry.clearField((Field)field, EntriesEventSource.SHARED));
            }
            if (remoteEntryMatchingOneLocalEntryFound) continue;
            entriesToInsertIntoLocalDatabase.add(idVersionEntry.getKey());
        }
        if (!entriesToInsertIntoLocalDatabase.isEmpty()) {
            this.bibDatabase.insertEntries(this.dbmsProcessor.partitionAndGetSharedEntries(entriesToInsertIntoLocalDatabase), EntriesEventSource.SHARED);
        }
    }

    private void removeNotSharedEntries(List<BibEntry> localEntries, Set<Integer> sharedIDs) {
        List<BibEntry> entriesToRemove = localEntries.stream().filter(localEntry -> !sharedIDs.contains(localEntry.getSharedBibEntryData().getSharedID())).collect(Collectors.toList());
        if (!entriesToRemove.isEmpty()) {
            this.eventBus.post((Object)new SharedEntriesNotPresentEvent(entriesToRemove));
            this.bibDatabase.removeEntries(entriesToRemove, EntriesEventSource.SHARED);
        }
    }

    @Override
    public void synchronizeSharedEntry(BibEntry bibEntry) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            BibDatabaseWriter.applySaveActions(bibEntry, this.metaData, this.fieldPreferences);
            this.dbmsProcessor.updateEntry(bibEntry);
        }
        catch (OfflineLockException exception) {
            this.eventBus.post((Object)new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error", (Throwable)e);
        }
    }

    public void synchronizeLocalMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.metaData.setEventPropagation(false);
            MetaDataParser parser = new MetaDataParser(this.fileMonitor);
            parser.parse(this.metaData, this.dbmsProcessor.getSharedMetaData(), this.keywordSeparator);
            this.metaData.setEventPropagation(true);
        }
        catch (ParseException e) {
            LOGGER.error("Parse error", (Throwable)e);
        }
    }

    private void synchronizeSharedMetaData(MetaData data, GlobalCitationKeyPatterns globalCiteKeyPattern) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.dbmsProcessor.setSharedMetaData(MetaDataSerializer.getSerializedStringMap(data, globalCiteKeyPattern));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", (Throwable)e);
        }
    }

    public void applyMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        for (BibEntry bibEntry : this.bibDatabase.getEntries()) {
            try {
                if (BibDatabaseWriter.applySaveActions(bibEntry, this.metaData, this.fieldPreferences).isEmpty()) continue;
                this.dbmsProcessor.updateEntry(bibEntry);
            }
            catch (OfflineLockException exception) {
                this.eventBus.post((Object)new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
            }
            catch (SQLException e) {
                LOGGER.error("SQL Error: ", (Throwable)e);
            }
        }
    }

    @Override
    public void pullChanges() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        this.pullWithLastEntry();
        this.synchronizeLocalDatabase();
        this.synchronizeLocalMetaData();
    }

    public void pullLastEntryChanges() {
        if (!this.lastEntryChanged.isEmpty()) {
            if (!this.checkCurrentConnection()) {
                return;
            }
            this.synchronizeLocalMetaData();
            this.pullWithLastEntry();
            this.synchronizeLocalDatabase();
        }
    }

    private void pullWithLastEntry() {
        if (!this.lastEntryChanged.isEmpty() && this.isPresentLocalBibEntry(this.lastEntryChanged.get())) {
            this.synchronizeSharedEntry(this.lastEntryChanged.get());
        }
        this.lastEntryChanged = Optional.empty();
    }

    public boolean checkCurrentConnection() {
        try {
            boolean isValid = this.currentConnection.isValid(0);
            if (!isValid) {
                LOGGER.warn("Lost SQL connection.");
                this.eventBus.post((Object)new ConnectionLostEvent(this.bibDatabaseContext));
            }
            return isValid;
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error during connection check", (Throwable)e);
            return false;
        }
    }

    public boolean isEventSourceAccepted(EntriesEvent event) {
        EntriesEventSource eventSource = event.getEntriesEventSource();
        return eventSource == EntriesEventSource.LOCAL || eventSource == EntriesEventSource.UNDO;
    }

    @Override
    public void openSharedDatabase(DatabaseConnection connection) throws DatabaseNotSupportedException {
        this.dbName = connection.getProperties().getDatabase();
        this.currentConnection = connection.getConnection();
        this.dbmsProcessor = DBMSProcessor.getProcessorInstance(connection);
        this.initializeDatabases();
    }

    @Override
    public void closeSharedDatabase() {
        this.pullLastEntryChanges();
        try {
            this.dbmsProcessor.stopNotificationListener();
            this.currentConnection.close();
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error:", (Throwable)e);
        }
    }

    private boolean isPresentLocalBibEntry(BibEntry bibEntry) {
        return this.bibDatabase.getEntries().contains((Object)bibEntry);
    }

    @Override
    public String getDBName() {
        return this.dbName;
    }

    public DBMSProcessor getDBProcessor() {
        return this.dbmsProcessor;
    }

    @Override
    public DatabaseConnectionProperties getConnectionProperties() {
        return this.dbmsProcessor.getDBMSConnectionProperties();
    }

    public void setMetaData(MetaData metaData) {
        this.metaData = metaData;
    }

    @Override
    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }
}

