/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui.linkedfile;

import com.tobiasdiez.easybind.EasyBind;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Optional;
import javafx.beans.property.DoubleProperty;
import javafx.beans.property.ReadOnlyDoubleProperty;
import javafx.beans.property.SimpleDoubleProperty;
import javafx.beans.value.ObservableValue;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLHandshakeException;
import javax.net.ssl.SSLSocketFactory;
import kong.unirest.core.UnirestException;
import org.jabref.gui.DialogService;
import org.jabref.gui.LibraryTab;
import org.jabref.gui.actions.SimpleCommand;
import org.jabref.gui.externalfiletype.ExternalFileType;
import org.jabref.gui.externalfiletype.ExternalFileTypes;
import org.jabref.gui.externalfiletype.StandardExternalFileType;
import org.jabref.gui.fieldeditors.LinkedFilesEditorViewModel;
import org.jabref.gui.fieldeditors.URLUtil;
import org.jabref.gui.util.BackgroundTask;
import org.jabref.gui.util.TaskExecutor;
import org.jabref.logic.externalfiles.LinkedFileHandler;
import org.jabref.logic.importer.FetcherClientException;
import org.jabref.logic.importer.FetcherServerException;
import org.jabref.logic.l10n.Localization;
import org.jabref.logic.net.ProgressInputStream;
import org.jabref.logic.net.URLDownload;
import org.jabref.logic.util.io.FileNameUniqueness;
import org.jabref.logic.util.io.FileUtil;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.LinkedFile;
import org.jabref.preferences.FilePreferences;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DownloadLinkedFileAction
extends SimpleCommand {
    private static final Logger LOGGER = LoggerFactory.getLogger(LibraryTab.class);
    private final DialogService dialogService;
    private final BibEntry entry;
    private final LinkedFile linkedFile;
    private final String suggestedName;
    private final String downloadUrl;
    private final FilePreferences filePreferences;
    private final TaskExecutor taskExecutor;
    private final boolean keepHtmlLink;
    private final BibDatabaseContext databaseContext;
    private final DoubleProperty downloadProgress = new SimpleDoubleProperty();
    private final LinkedFileHandler linkedFileHandler;

    public DownloadLinkedFileAction(BibDatabaseContext databaseContext, BibEntry entry, LinkedFile linkedFile, String downloadUrl, DialogService dialogService, FilePreferences filePreferences, TaskExecutor taskExecutor, String suggestedName, boolean keepHtmlLink) {
        this.databaseContext = databaseContext;
        this.entry = entry;
        this.linkedFile = linkedFile;
        this.suggestedName = suggestedName;
        this.downloadUrl = downloadUrl;
        this.dialogService = dialogService;
        this.filePreferences = filePreferences;
        this.taskExecutor = taskExecutor;
        this.keepHtmlLink = keepHtmlLink;
        this.linkedFileHandler = new LinkedFileHandler(linkedFile, entry, databaseContext, filePreferences);
    }

    public DownloadLinkedFileAction(BibDatabaseContext databaseContext, BibEntry entry, LinkedFile linkedFile, String downloadUrl, DialogService dialogService, FilePreferences filePreferences, TaskExecutor taskExecutor) {
        this(databaseContext, entry, linkedFile, downloadUrl, dialogService, filePreferences, taskExecutor, "", true);
    }

    public void execute() {
        LOGGER.info("Downloading file from {}", (Object)this.downloadUrl);
        if (this.downloadUrl.isEmpty() || !LinkedFile.isOnlineLink(this.downloadUrl)) {
            throw new UnsupportedOperationException("In order to download the file, the url has to be an online link");
        }
        Optional<Path> targetDirectory = this.databaseContext.getFirstExistingFileDir(this.filePreferences);
        if (targetDirectory.isEmpty()) {
            LOGGER.warn("File directory not available while downloading {}. Storing as URL in file field.", (Object)this.downloadUrl);
            return;
        }
        try {
            URLDownload urlDownload = new URLDownload(this.downloadUrl);
            if (!this.checkSSLHandshake(urlDownload)) {
                return;
            }
            this.doDownload(targetDirectory.get(), urlDownload);
        }
        catch (MalformedURLException exception) {
            this.dialogService.showErrorDialogAndWait(Localization.lang("Invalid URL", new Object[0]), exception);
        }
    }

    private void doDownload(Path targetDirectory, URLDownload urlDownload) {
        BackgroundTask<Path> downloadTask = this.prepareDownloadTask(targetDirectory, urlDownload);
        this.downloadProgress.bind((ObservableValue)downloadTask.workDonePercentageProperty());
        downloadTask.titleProperty().set((Object)Localization.lang("Downloading", new Object[0]));
        this.entry.getCitationKey().ifPresentOrElse(citationkey -> downloadTask.messageProperty().set((Object)Localization.lang("Fulltext for %0", citationkey)), () -> downloadTask.messageProperty().set((Object)Localization.lang("Fulltext for a new entry", new Object[0])));
        downloadTask.showToUser(true);
        downloadTask.onFailure(ex -> this.onFailure(urlDownload, (Exception)ex));
        downloadTask.onSuccess(destination -> this.onSuccess(targetDirectory, (Path)destination));
        this.taskExecutor.execute(downloadTask);
    }

    private void onSuccess(Path targetDirectory, Path downloadedFile) {
        boolean isDuplicate;
        assert (targetDirectory.isAbsolute());
        try {
            isDuplicate = FileNameUniqueness.isDuplicatedFile(targetDirectory, downloadedFile.getFileName(), this.dialogService);
        }
        catch (IOException e) {
            LOGGER.error("FileNameUniqueness.isDuplicatedFile failed", (Throwable)e);
            return;
        }
        if (isDuplicate) {
            LOGGER.info("File {} already exists in target directory {}.", (Object)downloadedFile.getFileName(), (Object)targetDirectory);
            return;
        }
        LinkedFile newLinkedFile = LinkedFilesEditorViewModel.fromFile(downloadedFile, this.databaseContext.getFileDirectories(this.filePreferences), this.filePreferences);
        if (newLinkedFile.getDescription().isEmpty() && !this.linkedFile.getDescription().isEmpty()) {
            newLinkedFile.setDescription(this.linkedFile.getDescription());
        }
        if (this.linkedFile.getSourceUrl().isEmpty() && LinkedFile.isOnlineLink(this.linkedFile.getLink())) {
            newLinkedFile.setSourceURL(this.linkedFile.getLink());
        } else {
            newLinkedFile.setSourceURL(this.linkedFile.getSourceUrl());
        }
        boolean isHtml = newLinkedFile.getFileType().equals(StandardExternalFileType.URL.getName());
        if (isHtml) {
            if (this.keepHtmlLink) {
                this.dialogService.notify(Localization.lang("Download '%0' was a HTML file. Keeping URL.", this.downloadUrl));
            } else {
                this.dialogService.notify(Localization.lang("Download '%0' was a HTML file. Removed.", this.downloadUrl));
                ArrayList<LinkedFile> newFiles = new ArrayList<LinkedFile>(this.entry.getFiles());
                newFiles.remove(this.linkedFile);
                this.entry.setFiles(newFiles);
                try {
                    Files.delete(downloadedFile);
                }
                catch (IOException e) {
                    LOGGER.error("Could not delete downloaded file {}.", (Object)downloadedFile, (Object)e);
                }
            }
        } else {
            this.entry.replaceDownloadedFile(this.linkedFile.getLink(), newLinkedFile);
        }
    }

    private void onFailure(URLDownload urlDownload, Exception ex) {
        LOGGER.error("Error downloading from URL: {}", (Object)urlDownload, (Object)ex);
        String fetcherExceptionMessage = ex.getMessage();
        String failedTitle = Localization.lang("Failed to download from URL", new Object[0]);
        if (ex instanceof FetcherClientException) {
            FetcherClientException clientException = (FetcherClientException)ex;
            int statusCode = clientException.getStatusCode();
            if (statusCode == 401) {
                this.dialogService.showInformationDialogAndWait(failedTitle, Localization.lang("401 Unauthorized: Access Denied. You are not authorized to access this resource. Please check your credentials and try again. If you believe you should have access, please contact the administrator for assistance.\nURL: %0 \n %1", urlDownload.getSource(), fetcherExceptionMessage));
            } else if (statusCode == 403) {
                this.dialogService.showInformationDialogAndWait(failedTitle, Localization.lang("403 Forbidden: Access Denied. You do not have permission to access this resource. Please contact the administrator for assistance or try a different action.\nURL: %0 \n %1", urlDownload.getSource(), fetcherExceptionMessage));
            } else if (statusCode == 404) {
                this.dialogService.showInformationDialogAndWait(failedTitle, Localization.lang("404 Not Found Error: The requested resource could not be found. It seems that the file you are trying to download is not available or has been moved. Please verify the URL and try again. If you believe this is an error, please contact the administrator for further assistance.\nURL: %0 \n %1", urlDownload.getSource(), fetcherExceptionMessage));
            }
        } else if (ex instanceof FetcherServerException) {
            FetcherServerException serverException = (FetcherServerException)ex;
            int statusCode = serverException.getStatusCode();
            this.dialogService.showInformationDialogAndWait(failedTitle, Localization.lang("Error downloading from URL. Cause is likely the server side. HTTP Error %0 \n %1 \nURL: %2 \nPlease try again later or contact the server administrator.", statusCode, fetcherExceptionMessage, urlDownload.getSource()));
        } else {
            this.dialogService.showErrorDialogAndWait(failedTitle, Localization.lang("Error message: %0 \nURL: %1 \nPlease check the URL and try again.", fetcherExceptionMessage, urlDownload.getSource()));
        }
    }

    private boolean checkSSLHandshake(URLDownload urlDownload) {
        try {
            urlDownload.canBeReached();
        }
        catch (UnirestException ex) {
            if (ex.getCause() instanceof SSLHandshakeException) {
                if (this.dialogService.showConfirmationDialogAndWait(Localization.lang("Download file", new Object[0]), Localization.lang("Unable to find valid certification path to requested target(%0), download anyway?", urlDownload.getSource().toString()))) {
                    return true;
                }
                this.dialogService.notify(Localization.lang("Download operation canceled.", new Object[0]));
                return false;
            }
            LOGGER.error("Error while checking if the file can be downloaded", (Throwable)ex);
            this.dialogService.notify(Localization.lang("Error downloading", new Object[0]));
            return false;
        }
        return true;
    }

    private BackgroundTask<Path> prepareDownloadTask(Path targetDirectory, URLDownload urlDownload) {
        SSLSocketFactory defaultSSLSocketFactory = HttpsURLConnection.getDefaultSSLSocketFactory();
        HostnameVerifier defaultHostnameVerifier = HttpsURLConnection.getDefaultHostnameVerifier();
        return BackgroundTask.wrap(() -> {
            String suggestedName;
            if (this.suggestedName.isEmpty()) {
                Optional<ExternalFileType> suggestedType = this.inferFileType(urlDownload);
                ExternalFileType externalFileType = suggestedType.orElse(StandardExternalFileType.PDF);
                suggestedName = this.linkedFileHandler.getSuggestedFileName(externalFileType.getExtension());
            } else {
                suggestedName = this.suggestedName;
            }
            String fulltextDir = FileUtil.createDirNameFromPattern(this.databaseContext.getDatabase(), this.entry, this.filePreferences.getFileDirectoryPattern());
            suggestedName = FileNameUniqueness.getNonOverWritingFileName(targetDirectory.resolve(fulltextDir), suggestedName);
            return targetDirectory.resolve(fulltextDir).resolve(suggestedName);
        }).then(destination -> new FileDownloadTask(urlDownload.getSource(), (Path)destination)).onFailure(ex -> LOGGER.error("Error in download", (Throwable)ex)).onFinished(() -> {
            URLDownload.setSSLVerification(defaultSSLSocketFactory, defaultHostnameVerifier);
            this.downloadProgress.unbind();
            this.downloadProgress.set(1.0);
        });
    }

    private Optional<ExternalFileType> inferFileType(URLDownload urlDownload) {
        Optional<ExternalFileType> suggestedType = this.inferFileTypeFromMimeType(urlDownload);
        if (suggestedType.isEmpty()) {
            suggestedType = this.inferFileTypeFromURL(urlDownload.getSource().toExternalForm());
        }
        return suggestedType;
    }

    private Optional<ExternalFileType> inferFileTypeFromMimeType(URLDownload urlDownload) {
        String mimeType = urlDownload.getMimeType();
        if (mimeType != null) {
            LOGGER.debug("MIME Type suggested: {}", (Object)mimeType);
            return ExternalFileTypes.getExternalFileTypeByMimeType(mimeType, this.filePreferences);
        }
        return Optional.empty();
    }

    private Optional<ExternalFileType> inferFileTypeFromURL(String url) {
        return URLUtil.getSuffix(url, this.filePreferences).flatMap(extension -> ExternalFileTypes.getExternalFileTypeByExt(extension, this.filePreferences));
    }

    public ReadOnlyDoubleProperty downloadProgress() {
        return this.downloadProgress;
    }

    private static class FileDownloadTask
    extends BackgroundTask<Path> {
        private final URL source;
        private final Path destination;

        public FileDownloadTask(URL source, Path destination) {
            this.source = source;
            this.destination = destination;
        }

        @Override
        protected Path call() throws Exception {
            URLDownload download = new URLDownload(this.source);
            try (ProgressInputStream inputStream = download.asInputStream();){
                EasyBind.subscribe((ObservableValue)inputStream.totalNumBytesReadProperty(), bytesRead -> this.updateProgress(bytesRead.longValue(), inputStream.getMaxNumBytes()));
                Files.createDirectories(this.destination.getParent(), new FileAttribute[0]);
                Files.copy(inputStream, this.destination, StandardCopyOption.REPLACE_EXISTING);
            }
            return this.destination;
        }
    }
}

