/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.gui.autosaveandbackup;

import com.google.common.eventbus.Subscribe;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import javafx.collections.ObservableList;
import org.jabref.gui.LibraryTab;
import org.jabref.gui.maintable.columns.MainTableColumn;
import org.jabref.logic.bibtex.InvalidFieldValueException;
import org.jabref.logic.exporter.AtomicFileWriter;
import org.jabref.logic.exporter.BibWriter;
import org.jabref.logic.exporter.BibtexDatabaseWriter;
import org.jabref.logic.exporter.SelfContainedSaveConfiguration;
import org.jabref.logic.util.BackupFileType;
import org.jabref.logic.util.CoarseChangeFilter;
import org.jabref.logic.util.io.BackupFileUtil;
import org.jabref.model.database.BibDatabase;
import org.jabref.model.database.BibDatabaseContext;
import org.jabref.model.database.event.BibDatabaseContextChangedEvent;
import org.jabref.model.entry.BibEntry;
import org.jabref.model.entry.BibEntryTypesManager;
import org.jabref.model.metadata.SaveOrder;
import org.jabref.model.metadata.SelfContainedSaveOrder;
import org.jabref.preferences.PreferencesService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(BackupManager.class);
    private static final int MAXIMUM_BACKUP_FILE_COUNT = 10;
    private static final int DELAY_BETWEEN_BACKUP_ATTEMPTS_IN_SECONDS = 19;
    private static Set<BackupManager> runningInstances = new HashSet<BackupManager>();
    private final BibDatabaseContext bibDatabaseContext;
    private final PreferencesService preferences;
    private final ScheduledThreadPoolExecutor executor;
    private final CoarseChangeFilter changeFilter;
    private final BibEntryTypesManager entryTypesManager;
    private final LibraryTab libraryTab;
    private final Queue<Path> backupFilesQueue = new LinkedBlockingQueue<Path>();
    private boolean needsBackup = false;

    BackupManager(LibraryTab libraryTab, BibDatabaseContext bibDatabaseContext, BibEntryTypesManager entryTypesManager, PreferencesService preferences) {
        this.bibDatabaseContext = bibDatabaseContext;
        this.entryTypesManager = entryTypesManager;
        this.preferences = preferences;
        this.executor = new ScheduledThreadPoolExecutor(2);
        this.libraryTab = libraryTab;
        this.changeFilter = new CoarseChangeFilter(bibDatabaseContext);
        this.changeFilter.registerListener(this);
    }

    static Path getBackupPathForNewBackup(Path originalPath, Path backupDir) {
        return BackupFileUtil.getPathForNewBackupFileAndCreateDirectory(originalPath, BackupFileType.BACKUP, backupDir);
    }

    static Optional<Path> getLatestBackupPath(Path originalPath, Path backupDir) {
        return BackupFileUtil.getPathOfLatestExistingBackupFile(originalPath, BackupFileType.BACKUP, backupDir);
    }

    public static BackupManager start(LibraryTab libraryTab, BibDatabaseContext bibDatabaseContext, BibEntryTypesManager entryTypesManager, PreferencesService preferences) {
        BackupManager backupManager = new BackupManager(libraryTab, bibDatabaseContext, entryTypesManager, preferences);
        backupManager.startBackupTask(preferences.getFilePreferences().getBackupDirectory());
        runningInstances.add(backupManager);
        return backupManager;
    }

    public static void discardBackup(BibDatabaseContext bibDatabaseContext, Path backupDir) {
        runningInstances.stream().filter(instance -> instance.bibDatabaseContext == bibDatabaseContext).forEach(backupManager -> backupManager.discardBackup(backupDir));
    }

    public static void shutdown(BibDatabaseContext bibDatabaseContext, Path backupDir, boolean createBackup) {
        runningInstances.stream().filter(instance -> instance.bibDatabaseContext == bibDatabaseContext).forEach(backupManager -> backupManager.shutdown(backupDir, createBackup));
        runningInstances.removeIf(instance -> instance.bibDatabaseContext == bibDatabaseContext);
    }

    public static boolean backupFileDiffers(Path originalPath, Path backupDir) {
        Path discardedFile = BackupManager.determineDiscardedFile(originalPath, backupDir);
        if (Files.exists(discardedFile, new LinkOption[0])) {
            try {
                Files.delete(discardedFile);
            }
            catch (IOException e) {
                LOGGER.error("Could not remove discarded file {}", (Object)discardedFile, (Object)e);
                return true;
            }
            return false;
        }
        return BackupManager.getLatestBackupPath(originalPath, backupDir).map(latestBackupPath -> {
            FileTime currentFileLastModifiedTime;
            FileTime latestBackupFileLastModifiedTime;
            try {
                latestBackupFileLastModifiedTime = Files.getLastModifiedTime(latestBackupPath, new LinkOption[0]);
            }
            catch (IOException e) {
                LOGGER.debug("Could not get timestamp of backup file {}", latestBackupPath, (Object)e);
                return false;
            }
            try {
                currentFileLastModifiedTime = Files.getLastModifiedTime(originalPath, new LinkOption[0]);
            }
            catch (IOException e) {
                LOGGER.debug("Could not get timestamp of current file file {}", (Object)originalPath, (Object)e);
                return false;
            }
            if (latestBackupFileLastModifiedTime.compareTo(currentFileLastModifiedTime) <= 0) {
                return false;
            }
            try {
                boolean result;
                boolean bl = result = Files.mismatch(originalPath, latestBackupPath) != -1L;
                if (result) {
                    LOGGER.info("Backup file {} differs from current file {}", latestBackupPath, (Object)originalPath);
                }
                return result;
            }
            catch (IOException e) {
                LOGGER.debug("Could not compare original file and backup file.", (Throwable)e);
                return true;
            }
        }).orElse(false);
    }

    public static void restoreBackup(Path originalPath, Path backupDir) {
        Optional<Path> backupPath = BackupManager.getLatestBackupPath(originalPath, backupDir);
        if (backupPath.isEmpty()) {
            LOGGER.error("There is no backup file");
            return;
        }
        try {
            Files.copy(backupPath.get(), originalPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            LOGGER.error("Error while restoring the backup file.", (Throwable)e);
        }
    }

    Optional<Path> determineBackupPathForNewBackup(Path backupDir) {
        return this.bibDatabaseContext.getDatabasePath().map(path -> BackupManager.getBackupPathForNewBackup(path, backupDir));
    }

    void performBackup(Path backupPath) {
        if (!this.needsBackup) {
            return;
        }
        while (this.backupFilesQueue.size() >= 10) {
            Path oldestBackupFile = this.backupFilesQueue.poll();
            try {
                Files.delete(oldestBackupFile);
            }
            catch (IOException e) {
                LOGGER.error("Could not delete backup file {}", (Object)oldestBackupFile, (Object)e);
            }
        }
        SelfContainedSaveOrder saveOrder = this.bibDatabaseContext.getMetaData().getSaveOrder().map(so -> {
            if (so.getOrderType() == SaveOrder.OrderType.TABLE) {
                ObservableList sortOrder = this.libraryTab.getMainTable().getSortOrder();
                return new SelfContainedSaveOrder(SaveOrder.OrderType.SPECIFIED, sortOrder.stream().filter(col -> col instanceof MainTableColumn).map(column -> ((MainTableColumn)((Object)((Object)column))).getModel()).flatMap(model -> model.getSortCriteria().stream()).toList());
            }
            return SelfContainedSaveOrder.of(so);
        }).orElse(SaveOrder.getDefaultSaveOrder());
        SelfContainedSaveConfiguration saveConfiguration = (SelfContainedSaveConfiguration)new SelfContainedSaveConfiguration().withMakeBackup(false).withSaveOrder(saveOrder).withReformatOnSave(this.preferences.getLibraryPreferences().shouldAlwaysReformatOnSave());
        List<BibEntry> list = this.bibDatabaseContext.getDatabase().getEntries().stream().map(BibEntry::clone).map(BibEntry.class::cast).toList();
        BibDatabase bibDatabaseClone = new BibDatabase(list);
        BibDatabaseContext bibDatabaseContextClone = new BibDatabaseContext(bibDatabaseClone, this.bibDatabaseContext.getMetaData());
        Charset encoding = this.bibDatabaseContext.getMetaData().getEncoding().orElse(StandardCharsets.UTF_8);
        try (AtomicFileWriter writer = new AtomicFileWriter(backupPath, encoding, false);){
            BibWriter bibWriter = new BibWriter(writer, this.bibDatabaseContext.getDatabase().getNewLineSeparator());
            new BibtexDatabaseWriter(bibWriter, saveConfiguration, this.preferences.getFieldPreferences(), this.preferences.getCitationKeyPatternPreferences(), this.entryTypesManager).saveDatabase(bibDatabaseContextClone);
            this.backupFilesQueue.add(backupPath);
            this.needsBackup = false;
        }
        catch (IOException e) {
            this.logIfCritical(backupPath, e);
        }
    }

    private static Path determineDiscardedFile(Path file, Path backupDir) {
        return backupDir.resolve(BackupFileUtil.getUniqueFilePrefix(file) + "--" + String.valueOf(file.getFileName()) + "--discarded");
    }

    public void discardBackup(Path backupDir) {
        Path path = BackupManager.determineDiscardedFile(this.bibDatabaseContext.getDatabasePath().get(), backupDir);
        try {
            Files.createFile(path, new FileAttribute[0]);
        }
        catch (IOException e) {
            LOGGER.info("Could not create backup file {}", (Object)path, (Object)e);
        }
    }

    private void logIfCritical(Path backupPath, IOException e) {
        Throwable innermostCause = e;
        while (innermostCause.getCause() != null) {
            innermostCause = innermostCause.getCause();
        }
        boolean isErrorInField = innermostCause instanceof InvalidFieldValueException;
        if (!isErrorInField) {
            LOGGER.error("Error while saving to file {}", (Object)backupPath, (Object)e);
        }
    }

    @Subscribe
    public synchronized void listen(BibDatabaseContextChangedEvent event) {
        if (!event.isFilteredOut()) {
            this.needsBackup = true;
        }
    }

    private void startBackupTask(Path backupDir) {
        this.fillQueue(backupDir);
        this.executor.scheduleAtFixedRate(() -> this.determineBackupPathForNewBackup(backupDir).ifPresent(path -> this.performBackup((Path)path)), 19L, 19L, TimeUnit.SECONDS);
    }

    private void fillQueue(Path backupDir) {
        if (!Files.exists(backupDir, new LinkOption[0])) {
            return;
        }
        this.bibDatabaseContext.getDatabasePath().ifPresent(databasePath -> {
            String prefix = BackupFileUtil.getUniqueFilePrefix(databasePath) + "--" + String.valueOf(databasePath.getFileName());
            try {
                List<Path> allSavFiles = Files.list(backupDir).filter(p -> p.getFileName().toString().startsWith(prefix)).sorted().toList();
                this.backupFilesQueue.addAll(allSavFiles);
            }
            catch (IOException e) {
                LOGGER.error("Could not determine most recent file", (Throwable)e);
            }
        });
    }

    private void shutdown(Path backupDir, boolean createBackup) {
        this.changeFilter.unregisterListener(this);
        this.changeFilter.shutdown();
        this.executor.shutdown();
        if (createBackup) {
            this.determineBackupPathForNewBackup(backupDir).ifPresent(this::performBackup);
        }
    }
}

